// vibe_mojo_interface.js
// JavaScript wrapper for Vibe AI Mojo interface

/**
 * Vibe AI Mojo Interface Wrapper
 * Provides JavaScript bindings to the C++ Vibe AI tools
 */
class VibeMojoInterface {
  constructor() {
    this.vibeAIService = null;
    this.keyManager = null;
    this.scriptManager = null;
    this.isInitialized = false;
  }

  /**
   * Initialize the Mojo interface
   */
  async initialize() {
    if (this.isInitialized) {
      return true;
    }

    try {
      // Import the generated Mojo bindings (tools only)
      const { BrowserNavigationTool, DOMManipulationTool, SearchTool, VibeAIKeyManager, VibeAIScriptManager } = await import("/chrome/browser/ai/vibe/vibe.mojom.js");
      
      // Create Mojo service connections for tools
      this.navigationTool = new BrowserNavigationTool();
      this.domTool = new DOMManipulationTool();
      this.searchTool = new SearchTool();
      this.keyManager = new VibeAIKeyManager();
      this.scriptManager = new VibeAIScriptManager();
      
      this.isInitialized = true;
      console.log("Vibe Mojo interface initialized successfully");
      return true;
    } catch (error) {
      console.error("Failed to initialize Vibe Mojo interface:", error);
      return false;
    }
  }

  // Note: AI processing removed - now handled directly in JavaScript
  // This interface now focuses only on browser tools and configuration

  /**
   * Execute a browser action using C++ tools
   * @param {Object} action - Browser action to execute
   * @returns {Promise<Object>} Action result
   */
  async executeBrowserAction(action) {
    if (!this.isInitialized) {
      await this.initialize();
    }

    try {
      const browserAction = {
        tool: action.tool,
        params_json: JSON.stringify(action.params || {}),
        context: action.context || null
      };

      const result = await this.vibeAIService.executeBrowserAction(browserAction);
      return result;
    } catch (error) {
      console.error("❌ Failed to execute browser action:", error);
      throw error;
    }
  }

  /**
   * Get current page context for better AI understanding
   * @returns {Promise<string>} Context JSON string
   */
  async getPageContext() {
    if (!this.isInitialized) {
      await this.initialize();
    }

    try {
      const context = await this.vibeAIService.getPageContext();
      return context;
    } catch (error) {
      console.error("❌ Failed to get page context:", error);
      throw error;
    }
  }

  /**
   * Get available tools/capabilities
   * @returns {Promise<Array<string>>} Array of tool names
   */
  async getAvailableTools() {
    if (!this.isInitialized) {
      await this.initialize();
    }

    try {
      const tools = await this.vibeAIService.getAvailableTools();
      return tools;
    } catch (error) {
      console.error("❌ Failed to get available tools:", error);
      throw error;
    }
  }

  /**
   * Set API key for a backend
   * @param {string} backend - Backend name (openai, claude, etc.)
   * @param {string} apiKey - API key
   * @returns {Promise<boolean>} Success status
   */
  async setApiKey(backend, apiKey) {
    if (!this.isInitialized) {
      await this.initialize();
    }

    try {
      const result = await this.keyManager.setApiKey(backend, apiKey);
      return result;
    } catch (error) {
      console.error("❌ Failed to set API key:", error);
      throw error;
    }
  }

  /**
   * Check if API key is configured for a backend
   * @param {string} backend - Backend name
   * @returns {Promise<boolean>} Whether API key is set
   */
  async hasApiKey(backend) {
    if (!this.isInitialized) {
      await this.initialize();
    }

    try {
      const result = await this.keyManager.hasApiKey(backend);
      return result;
    } catch (error) {
      console.error("❌ Failed to check API key:", error);
      throw error;
    }
  }

  /**
   * Validate API key for a backend
   * @param {string} backend - Backend name
   * @returns {Promise<Object>} Validation result
   */
  async validateApiKey(backend) {
    if (!this.isInitialized) {
      await this.initialize();
    }

    try {
      const result = await this.keyManager.validateApiKey(backend);
      return result;
    } catch (error) {
      console.error("❌ Failed to validate API key:", error);
      throw error;
    }
  }
}

// Export singleton instance
const vibeMojoInterface = new VibeMojoInterface();
export default vibeMojoInterface;