/**
 * Video Recorder Module for Puppeteer Tests
 * 
 * Creates animated GIFs from test screenshots using ffmpeg.
 * GIFs can be embedded inline in GitHub PR comments.
 * 
 * Usage:
 *   const { VideoRecorder } = require('./lib/utils/video-recorder');
 *   
 *   const recorder = new VideoRecorder(testDir);
 *   recorder.addScreenshot('/path/to/screenshot.png', 'step_name');
 *   await recorder.createVideo(); // Creates testDir/test-recording.gif
 * 
 * Features:
 *   - Collects screenshots with timestamps
 *   - Creates optimized GIF using ffmpeg with palette generation
 *   - Handles variable screenshot sizes (scales to consistent dimensions)
 *   - Limits screenshots to keep file size reasonable for GitHub
 */

import { spawnSync } from 'child_process';
import fs from 'fs';
import path from 'path';

/**
 * Check if ffmpeg is available
 * @returns {boolean}
 */
export function hasFfmpeg() {
  try {
    const result = spawnSync('which', ['ffmpeg'], { stdio: 'pipe' });
    return result.status === 0;
  } catch {
    return false;
  }
}

/**
 * Video Recorder class for creating test GIFs from screenshots
 */
export class VideoRecorder {
  /**
   * @param {string} testDir - Directory where test outputs are stored
   * @param {object} options - Configuration options
   * @param {number} options.width - Output GIF width (default: 1200)
   * @param {number} options.height - Output GIF height (default: 800)
   * @param {number} options.fps - Frames per second (default: 1 for GIF - 1 sec per frame)
   * @param {number} options.secondsPerFrame - Seconds per frame (alternative to fps)
   * @param {number} options.maxScreenshots - Max screenshots to include (default: 30)
   * @param {string} options.outputName - Output filename (default: 'test-recording.gif')
   */
  constructor(testDir, options = {}) {
    this.testDir = testDir;
    this.screenshotsDir = path.join(testDir, 'screenshots');
    this.framesDir = path.join(testDir, 'gif-frames');
    
    // Convert secondsPerFrame to fps if provided
    let fps = options.fps || 1;  // Default: 1 fps = 1 second per frame
    if (options.secondsPerFrame) {
      fps = 1 / options.secondsPerFrame;
    }
    
    this.options = {
      width: options.width || 1200,
      height: options.height || 800,
      fps: fps,
      maxScreenshots: options.maxScreenshots || 30,
      outputName: options.outputName || 'test-recording.gif'
    };
    
    this.screenshots = [];
    this.frameCount = 0;
  }

  /**
   * Add a screenshot to the video queue
   * @param {string} screenshotPath - Path to the screenshot file
   * @param {string} name - Name/label for this screenshot
   * @param {object} metadata - Optional metadata (timestamp, scenario, etc.)
   */
  addScreenshot(screenshotPath, name, metadata = {}) {
    this.screenshots.push({
      path: screenshotPath,
      name,
      timestamp: metadata.timestamp || Date.now(),
      scenario: metadata.scenario || null,
      ...metadata
    });
  }

  /**
   * Collect all screenshots from the screenshots directory
   * Automatically finds and sorts screenshots by filename
   * @param {string} pattern - Optional glob pattern to filter screenshots
   */
  collectScreenshots(pattern = '*.png') {
    if (!fs.existsSync(this.screenshotsDir)) {
      console.warn(`Screenshots directory not found: ${this.screenshotsDir}`);
      return;
    }

    const files = fs.readdirSync(this.screenshotsDir)
      .filter(f => f.endsWith('.png'))
      .sort();

    for (const file of files) {
      const filePath = path.join(this.screenshotsDir, file);
      const stats = fs.statSync(filePath);
      
      // Extract name from filename (remove extension and leading numbers)
      const name = file.replace(/^\d+_/, '').replace(/\.png$/, '');
      
      this.addScreenshot(filePath, name, {
        timestamp: stats.mtime.getTime()
      });
    }

    console.log(`Collected ${this.screenshots.length} screenshots`);
  }

  /**
   * Collect paired screenshots (home + sidepanel) from screenshots directory
   * Creates side-by-side combined frames
   * Now also includes sidepanel-only and active-tab-only screenshots to show complete test flow
   * @returns {Array} Array of screenshots (paired, sidepanel-only, or active-tab-only)
   */
  collectPairedScreenshots() {
    if (!fs.existsSync(this.screenshotsDir)) {
      console.warn(`Screenshots directory not found: ${this.screenshotsDir}`);
      return [];
    }

    const files = fs.readdirSync(this.screenshotsDir)
      .filter(f => f.endsWith('.png'))
      .sort();

    // Group files by step prefix
    const stepGroups = new Map();
    
    for (const file of files) {
      // Skip doc_ screenshots (documentation screenshots, not test steps)
      if (file.startsWith('doc_')) continue;
      
      // Extract step prefix (numeric or Google Workspace format)
      let prefix;
      const gwMatch = file.match(/^((?:G?\d+|[A-Za-z]+\d*)_(?:\d+ms|final|start))/);
      const numMatch = file.match(/^(\d+)_/);
      
      if (gwMatch) {
        prefix = gwMatch[1];
      } else if (numMatch) {
        prefix = numMatch[1];
      } else {
        continue;
      }
      
      if (!stepGroups.has(prefix)) {
        stepGroups.set(prefix, { sidepanel: null, activeTabs: [] });
      }
      
      const group = stepGroups.get(prefix);
      if (file.includes('sidepanel.html')) {
        group.sidepanel = file;
      } else {
        group.activeTabs.push(file);
      }
    }

    // Convert groups to screenshot entries, sorted by step number
    const entries = [];
    const sortedPrefixes = Array.from(stepGroups.keys()).sort((a, b) => {
      // Extract numeric part for sorting
      const numA = parseInt(a.match(/\d+/)?.[0] || '0');
      const numB = parseInt(b.match(/\d+/)?.[0] || '0');
      return numA - numB;
    });
    
    for (const prefix of sortedPrefixes) {
      const group = stepGroups.get(prefix);
      const activeTabFile = group.activeTabs[0]; // Use first active tab if multiple
      
      // Include entry even if only sidepanel or only active tab exists
      if (group.sidepanel || activeTabFile) {
        entries.push({
          home: activeTabFile ? path.join(this.screenshotsDir, activeTabFile) : null,
          sidepanel: group.sidepanel ? path.join(this.screenshotsDir, group.sidepanel) : null,
          timestamp: prefix,
          name: (activeTabFile || group.sidepanel).replace(/\.png$/, ''),
          type: group.sidepanel && activeTabFile ? 'paired' : (group.sidepanel ? 'sidepanel-only' : 'active-only')
        });
      }
    }

    const paired = entries.filter(e => e.type === 'paired').length;
    const sidepanelOnly = entries.filter(e => e.type === 'sidepanel-only').length;
    const activeOnly = entries.filter(e => e.type === 'active-only').length;
    console.log(`Collected ${entries.length} screenshots: ${paired} paired, ${sidepanelOnly} sidepanel-only, ${activeOnly} active-only`);
    return entries;
  }

  /**
   * Sample screenshots evenly if we have more than maxScreenshots
   * @returns {Array} Sampled screenshot array
   */
  sampleScreenshots() {
    if (this.screenshots.length <= this.options.maxScreenshots) {
      return this.screenshots;
    }

    // Always include first and last
    const sampled = [this.screenshots[0]];
    const step = (this.screenshots.length - 1) / (this.options.maxScreenshots - 1);
    
    for (let i = 1; i < this.options.maxScreenshots - 1; i++) {
      const index = Math.round(i * step);
      sampled.push(this.screenshots[index]);
    }
    
    sampled.push(this.screenshots[this.screenshots.length - 1]);
    
    console.log(`Sampled ${sampled.length} from ${this.screenshots.length} screenshots`);
    return sampled;
  }

  /**
   * Create GIF from collected screenshots
   * @param {object} options - Creation options
   * @param {boolean} options.sideBySide - Create side-by-side paired screenshots (home + sidepanel)
   * @returns {Promise<{success: boolean, videoPath?: string, error?: string}>}
   */
  async createVideo(options = {}) {
    if (!hasFfmpeg()) {
      return {
        success: false,
        error: 'ffmpeg not found. Install with: brew install ffmpeg'
      };
    }

    // If side-by-side mode, collect paired screenshots
    let pairedScreenshots = [];
    if (options.sideBySide) {
      pairedScreenshots = this.collectPairedScreenshots();
      if (pairedScreenshots.length === 0) {
        console.log('No paired screenshots found, falling back to regular screenshots');
        options.sideBySide = false;
      }
    }

    if (!options.sideBySide) {
      if (this.screenshots.length === 0) {
        // Try to collect screenshots automatically
        this.collectScreenshots();
      }

      if (this.screenshots.length === 0) {
        return {
          success: false,
          error: 'No screenshots to create GIF from'
        };
      }
    }

    // Clean and create frames directory
    if (fs.existsSync(this.framesDir)) {
      fs.rmSync(this.framesDir, { recursive: true });
    }
    fs.mkdirSync(this.framesDir, { recursive: true });

    if (options.sideBySide) {
      // Create frames from screenshots (paired, sidepanel-only, or active-only)
      const SIDEPANEL_WIDTH = 362;
      const outputWidth = this.options.width || 1200;
      const outputHeight = this.options.height || 700;

      for (const entry of pairedScreenshots) {
        const framePath = path.join(this.framesDir, `frame_${String(this.frameCount).padStart(5, '0')}.png`);
        let frameCreated = false;

        if (entry.type === 'paired') {
          // Both home and sidepanel exist - create side-by-side composite
          if (!fs.existsSync(entry.home) || !fs.existsSync(entry.sidepanel)) {
            console.warn(`Paired screenshot files not found: ${entry.timestamp}`);
            continue;
          }

          const mergeResult = spawnSync('ffmpeg', [
            '-y',
            '-i', entry.home,
            '-i', entry.sidepanel,
            '-filter_complex',
            `[0:v]scale=${outputWidth - SIDEPANEL_WIDTH}:${outputHeight}:force_original_aspect_ratio=decrease,pad=${outputWidth - SIDEPANEL_WIDTH}:${outputHeight}:(ow-iw)/2:(oh-ih)/2:white[left];` +
            `[1:v]scale=${SIDEPANEL_WIDTH}:${outputHeight}:force_original_aspect_ratio=decrease,pad=${SIDEPANEL_WIDTH}:${outputHeight}:(ow-iw)/2:(oh-ih)/2:white[right];` +
            `[left][right]hstack=inputs=2`,
            '-frames:v', '1',
            framePath
          ], { stdio: 'pipe' });

          if (mergeResult.status === 0) {
            frameCreated = true;
          } else {
            console.warn(`Failed to create side-by-side frame: ${mergeResult.stderr?.toString()}`);
          }
        } else if (entry.type === 'sidepanel-only') {
          // Only sidepanel exists - show sidepanel on right, gray placeholder on left
          if (!fs.existsSync(entry.sidepanel)) {
            console.warn(`Sidepanel screenshot not found: ${entry.timestamp}`);
            continue;
          }

          // Create sidepanel-only frame with gray left placeholder (simulates hidden active tab)
          const sidepanelResult = spawnSync('ffmpeg', [
            '-y',
            '-f', 'lavfi',
            '-i', `color=c=#e8e8e8:s=${outputWidth - SIDEPANEL_WIDTH}x${outputHeight}`,
            '-i', entry.sidepanel,
            '-filter_complex',
            `[1:v]scale=${SIDEPANEL_WIDTH}:${outputHeight}:force_original_aspect_ratio=decrease,pad=${SIDEPANEL_WIDTH}:${outputHeight}:(ow-iw)/2:(oh-ih)/2:white[right];` +
            `[0:v][right]hstack=inputs=2`,
            '-frames:v', '1',
            framePath
          ], { stdio: 'pipe' });

          if (sidepanelResult.status === 0) {
            frameCreated = true;
          } else {
            // Fallback: just scale sidepanel to full width
            const scaleResult = spawnSync('ffmpeg', [
              '-y',
              '-i', entry.sidepanel,
              '-vf', `scale=${outputWidth}:${outputHeight}:force_original_aspect_ratio=decrease,pad=${outputWidth}:${outputHeight}:(ow-iw)/2:(oh-ih)/2:white`,
              '-frames:v', '1',
              framePath
            ], { stdio: 'pipe' });
            frameCreated = scaleResult.status === 0;
          }
        } else if (entry.type === 'active-only') {
          // Only active tab exists - scale to fit full width
          if (!fs.existsSync(entry.home)) {
            console.warn(`Active tab screenshot not found: ${entry.timestamp}`);
            continue;
          }

          const scaleResult = spawnSync('ffmpeg', [
            '-y',
            '-i', entry.home,
            '-vf', `scale=${outputWidth}:${outputHeight}:force_original_aspect_ratio=decrease,pad=${outputWidth}:${outputHeight}:(ow-iw)/2:(oh-ih)/2:white`,
            '-frames:v', '1',
            framePath
          ], { stdio: 'pipe' });

          if (scaleResult.status === 0) {
            frameCreated = true;
          } else {
            // Last resort: copy file directly
            try {
              fs.copyFileSync(entry.home, framePath);
              frameCreated = true;
            } catch (e) {
              console.warn(`Failed to copy active tab screenshot: ${e.message}`);
            }
          }
        }

        if (frameCreated) {
          this.frameCount++;
        }
      }

      console.log(`Created ${this.frameCount} frames from ${pairedScreenshots.length} screenshots`);
    } else {
      // Sample screenshots to keep GIF size reasonable
      const screenshotsToUse = this.sampleScreenshots();

      // Process each screenshot into frames
      for (const screenshot of screenshotsToUse) {
        if (!fs.existsSync(screenshot.path)) {
          console.warn(`Screenshot not found: ${screenshot.path}`);
          continue;
        }

        // Scale screenshot to consistent dimensions
        // For tall screenshots (full page), crop to top viewport instead of squishing
        const framePath = path.join(this.framesDir, `frame_${String(this.frameCount).padStart(5, '0')}.png`);

        const scaleResult = spawnSync('ffmpeg', [
          '-y',
          '-i', screenshot.path,
          '-vf', `scale=${this.options.width}:-1,crop=${this.options.width}:${this.options.height}:0:0,pad=${this.options.width}:${this.options.height}:(ow-iw)/2:(oh-ih)/2:white`,
          '-frames:v', '1',
          framePath
        ], { stdio: 'pipe' });

        if (scaleResult.status !== 0) {
          // Fallback: just copy the file
          fs.copyFileSync(screenshot.path, framePath);
        }

        this.frameCount++;
      }

      console.log(`Created ${this.frameCount} frames from ${screenshotsToUse.length} screenshots`);
    }

    // Create GIF from frames using two-pass palette method for better quality
    const outputPath = path.join(this.testDir, this.options.outputName);
    const palettePath = path.join(this.framesDir, 'palette.png');
    
    // Pass 1: Generate palette
    const paletteResult = spawnSync('ffmpeg', [
      '-y',
      '-framerate', String(this.options.fps),
      '-i', path.join(this.framesDir, 'frame_%05d.png'),
      '-vf', 'palettegen=stats_mode=diff',
      palettePath
    ], { stdio: 'pipe' });

    if (paletteResult.status !== 0) {
      // Fallback to single-pass GIF creation
      console.log('Palette generation failed, using single-pass');
      const simpleResult = spawnSync('ffmpeg', [
        '-y',
        '-framerate', String(this.options.fps),
        '-i', path.join(this.framesDir, 'frame_%05d.png'),
        '-vf', `scale=${this.options.width}:-1:flags=lanczos`,
        outputPath
      ], { stdio: 'pipe' });

      if (simpleResult.status !== 0) {
        const stderr = simpleResult.stderr?.toString() || 'Unknown error';
        // Clean up frames directory
        try { fs.rmSync(this.framesDir, { recursive: true }); } catch (e) {}
        return {
          success: false,
          error: `ffmpeg failed: ${stderr.substring(0, 500)}`
        };
      }
    } else {
      // Pass 2: Create GIF using palette
      const gifResult = spawnSync('ffmpeg', [
        '-y',
        '-framerate', String(this.options.fps),
        '-i', path.join(this.framesDir, 'frame_%05d.png'),
        '-i', palettePath,
        '-lavfi', 'paletteuse=dither=bayer:bayer_scale=5:diff_mode=rectangle',
        outputPath
      ], { stdio: 'pipe' });

      if (gifResult.status !== 0) {
        const stderr = gifResult.stderr?.toString() || 'Unknown error';
        // Clean up frames directory
        try { fs.rmSync(this.framesDir, { recursive: true }); } catch (e) {}
        return {
          success: false,
          error: `ffmpeg GIF creation failed: ${stderr.substring(0, 500)}`
        };
      }
    }

    // Clean up frames directory
    try {
      fs.rmSync(this.framesDir, { recursive: true });
    } catch (e) {
      // Ignore cleanup errors
    }

    // Get GIF info
    const stats = fs.statSync(outputPath);
    const sizeMB = (stats.size / 1024 / 1024).toFixed(2);
    const durationSec = this.frameCount / this.options.fps;

    console.log(`GIF created: ${outputPath}`);
    console.log(`  Size: ${sizeMB} MB`);
    console.log(`  Duration: ${durationSec} seconds`);
    console.log(`  Frames: ${this.frameCount}`);

    return {
      success: true,
      videoPath: outputPath,
      size: stats.size,
      sizeMB: parseFloat(sizeMB),
      duration: durationSec,
      frameCount: this.frameCount,
      screenshotCount: this.screenshots.length
    };
  }

  /**
   * Create GIF from a test directory's screenshots
   * Static helper for quick GIF creation
   * 
   * @param {string} testDir - Test directory path
   * @param {object} options - GIF options
   * @param {boolean} options.sideBySide - Create side-by-side paired screenshots (home + sidepanel)
   * @returns {Promise<{success: boolean, videoPath?: string, error?: string}>}
   */
  static async fromTestDirectory(testDir, options = {}) {
    const recorder = new VideoRecorder(testDir, options);
    if (!options.sideBySide) {
      recorder.collectScreenshots();
    }
    return recorder.createVideo(options);
  }
}

/**
 * Create GIFs for all recent test directories
 * Useful for batch processing after test runs
 * 
 * @param {string} testBaseDir - Base test directory (default: .test)
 * @param {number} maxDirs - Maximum number of directories to process (default: 5)
 * @returns {Promise<Array<{testDir: string, success: boolean, videoPath?: string}>>}
 */
export async function createVideosForRecentTests(testBaseDir = '.test', maxDirs = 5) {
  if (!fs.existsSync(testBaseDir)) {
    console.error(`Test base directory not found: ${testBaseDir}`);
    return [];
  }

  const dirs = fs.readdirSync(testBaseDir)
    .map(d => ({
      name: d,
      path: path.join(testBaseDir, d),
      mtime: fs.statSync(path.join(testBaseDir, d)).mtime.getTime()
    }))
    .filter(d => fs.statSync(d.path).isDirectory())
    .sort((a, b) => b.mtime - a.mtime)
    .slice(0, maxDirs);

  const results = [];

  for (const dir of dirs) {
    console.log(`\nProcessing: ${dir.name}`);
    
    const result = await VideoRecorder.fromTestDirectory(dir.path);
    results.push({
      testDir: dir.path,
      testName: dir.name,
      ...result
    });
  }

  return results;
}

// CLI support
if (process.argv[1]?.endsWith('video-recorder.js')) {
  const testDir = process.argv[2];
  
  if (!testDir) {
    console.log(`
Video Recorder - Create GIFs from test screenshots

Usage:
  node lib/utils/video-recorder.js <test-dir>
  node lib/utils/video-recorder.js .test/ExtensionMock-2024-01-24

Options (via environment):
  VIDEO_WIDTH=800      Output GIF width
  VIDEO_HEIGHT=500     Output GIF height
  VIDEO_FPS=0.5        Frames per second (0.5 = 2 sec per frame)
  VIDEO_MAX=30         Max screenshots to include
`);
    process.exit(0);
  }

  const options = {
    width: parseInt(process.env.VIDEO_WIDTH) || 800,
    height: parseInt(process.env.VIDEO_HEIGHT) || 500,
    fps: parseFloat(process.env.VIDEO_FPS) || 0.5,
    maxScreenshots: parseInt(process.env.VIDEO_MAX) || 30
  };

  VideoRecorder.fromTestDirectory(testDir, options)
    .then(result => {
      if (result.success) {
        console.log('\nSuccess!');
        console.log(`GIF: ${result.videoPath}`);
        process.exit(0);
      } else {
        console.error('\nFailed:', result.error);
        process.exit(1);
      }
    })
    .catch(err => {
      console.error('Error:', err.message);
      process.exit(1);
    });
}

export default VideoRecorder;
