/**
 * Shared utilities for detecting model capabilities
 * Used across both frontend (React) and backend (agent) code
 */

/**
 * Detect if a model supports vision based on model name
 * @param {string} modelId - Full model ID (e.g., "openrouter:openai/gpt-4o")
 * @returns {boolean} True if model supports vision
 */
export function detectVisionFromName(modelId) {
  if (!modelId) return false;
  
  const id = modelId.toLowerCase();
  return (
    id.includes('gpt-4o') ||
    id.includes('gpt-4-vision') ||
    id.includes('gpt-5') ||
    id.includes('o1') ||
    id.includes('o3') ||
    id.includes('o4') ||
    id.includes('claude-') ||
    id.includes('gemini') ||
    (id.includes('qwen') && id.includes('vl')) ||
    id.includes('pixtral')
  );
}

/**
 * Get cached model data from Chrome storage
 * @returns {Promise<Map<string, {supportsVision: boolean}> | null>} Map of model ID to capabilities
 */
async function getCachedModelData() {
  if (typeof chrome === 'undefined' || !chrome.storage?.local) {
    return null;
  }

  try {
    const result = await chrome.storage.local.get(['vibeCachedModels', 'vibeCachedModelsTimestamp']);
    const models = result.vibeCachedModels;
    const timestamp = result.vibeCachedModelsTimestamp;

    if (!models || !timestamp) {
      return null;
    }

    // Check if cache is still valid (24 hours)
    const age = Date.now() - timestamp;
    const maxAge = 24 * 60 * 60 * 1000;
    if (age > maxAge) {
      return null;
    }

    // Create map of model ID to capabilities
    const modelMap = new Map();
    for (const model of models) {
      modelMap.set(model.id, {
        supportsVision: model.supportsVision,
        supportsTools: model.supportsTools,
        context: model.context
      });
    }
    return modelMap;
  } catch (error) {
    console.error('Failed to get cached model data:', error);
    return null;
  }
}

/**
 * Detect if a model supports vision, using cached API data if available
 * This is the recommended function to use - it checks OpenRouter API cache first,
 * then falls back to name-based detection.
 * 
 * @param {string} modelId - Full model ID (e.g., "openrouter:openai/gpt-4o")
 * @returns {Promise<boolean>} True if model supports vision
 */
export async function detectModelSupportsVision(modelId) {
  if (!modelId) return false;

  // Try to get from cached model data first (OpenRouter API)
  const cachedModels = await getCachedModelData();
  if (cachedModels) {
    const modelInfo = cachedModels.get(modelId);
    if (modelInfo && modelInfo.supportsVision !== undefined) {
      return modelInfo.supportsVision;
    }
  }

  // Fallback to name-based detection
  return detectVisionFromName(modelId);
}
