/**
 * Langfuse Configuration and Integration
 * Provides LangChain integration for observability and debugging
 */

import { Langfuse } from 'langfuse';
import { CallbackHandler } from 'langfuse-langchain';

let langfuseClient = null;
let langfuseCallbackHandler = null;

/**
 * Initialize Langfuse client and callback handler
 * @param {Object} config - Configuration object
 * @param {string} config.publicKey - Langfuse public key
 * @param {string} config.secretKey - Langfuse secret key
 * @param {string} config.baseUrl - Langfuse base URL (optional, defaults to cloud)
 * @param {boolean} config.enabled - Whether Langfuse is enabled
 * @returns {Object} Initialized Langfuse client and callback handler
 */
export function initLangfuse(config = {}) {
  const {
    publicKey = process.env.LANGFUSE_PUBLIC_KEY,
    secretKey = process.env.LANGFUSE_SECRET_KEY,
    baseUrl = process.env.LANGFUSE_BASE_URL || 'https://cloud.langfuse.com',
    enabled = true
  } = config;

  // If not enabled or missing credentials, return null handlers
  if (!enabled || !publicKey || !secretKey) {
    console.log('[Langfuse] Disabled or missing credentials');
    return {
      client: null,
      callbackHandler: null,
      enabled: false
    };
  }

  try {
    // Initialize Langfuse client
    langfuseClient = new Langfuse({
      publicKey,
      secretKey,
      baseUrl,
      flushAt: 1, // Flush immediately for real-time observability
      flushInterval: 1000, // Flush every second
    });

    // Initialize LangChain callback handler
    langfuseCallbackHandler = new CallbackHandler({
      publicKey,
      secretKey,
      baseUrl,
      flushAt: 1,
      flushInterval: 1000,
    });

    console.log(`[Langfuse] Initialized successfully (${baseUrl})`);

    return {
      client: langfuseClient,
      callbackHandler: langfuseCallbackHandler,
      enabled: true
    };
  } catch (error) {
    console.error('[Langfuse] Initialization failed:', error.message);
    return {
      client: null,
      callbackHandler: null,
      enabled: false
    };
  }
}

/**
 * Get the current Langfuse callback handler
 * @returns {CallbackHandler|null} The callback handler or null if not initialized
 */
export function getLangfuseCallbackHandler() {
  return langfuseCallbackHandler;
}

/**
 * Get the current Langfuse client
 * @returns {Langfuse|null} The client or null if not initialized
 */
export function getLangfuseClient() {
  return langfuseClient;
}

/**
 * Flush any pending Langfuse events
 * Call this before the application exits
 */
export async function flushLangfuse() {
  if (langfuseClient) {
    await langfuseClient.flushAsync();
    console.log('[Langfuse] Flushed pending events');
  }
}

/**
 * Create a trace for manual tracking
 * @param {string} name - Trace name
 * @param {Object} metadata - Additional metadata
 * @returns {Object} Trace object
 */
export function createTrace(name, metadata = {}) {
  if (!langfuseClient) {
    return null;
  }

  return langfuseClient.trace({
    name,
    metadata,
    timestamp: new Date()
  });
}

/**
 * Shutdown Langfuse client
 */
export async function shutdownLangfuse() {
  if (langfuseClient) {
    await langfuseClient.shutdownAsync();
    langfuseClient = null;
    langfuseCallbackHandler = null;
    console.log('[Langfuse] Shutdown complete');
  }
}
