/**
 * Custom initChatModel that properly initializes chat models for different providers
 * This fixes the issue with ConfigurableModel not working properly with withStructuredOutput
 */

import { ChatOpenAI } from "@langchain/openai";
import { AzureChatOpenAI } from "@langchain/openai";
import { ChatGoogleGenerativeAI } from "@langchain/google-genai";
import { ChatAnthropic } from "@langchain/anthropic";
import { LLMWithRetry } from "./LLMWithRetry.js";
import { getLangfuseCallbackHandler } from "./langfuse-config.js";
import { ChromeNanoLLM } from "../providers/ChromeNanoLLM.js";

/**
 * Wrapper for LLMs that need special tool calling configuration.
 * Used for DeepSeek models which have issues with parallel tool calls.
 * Reference: https://github.com/langchain-ai/langchainjs/issues/7815
 */
class DeepSeekLLMWrapper {
  constructor(llm) {
    this.llm = llm;
    // Proxy all properties/methods to the underlying LLM
    return new Proxy(this, {
      get(target, prop, receiver) {
        // Override bindTools to always disable parallel_tool_calls
        if (prop === 'bindTools') {
          return (tools, kwargs = {}) => {
            console.log('[DeepSeek] Binding tools with parallel_tool_calls: false');
            // Force parallel_tool_calls to false for DeepSeek
            const deepseekKwargs = {
              ...kwargs,
              parallel_tool_calls: false,
            };
            return target.llm.bindTools(tools, deepseekKwargs);
          };
        }
        // For all other properties, delegate to the wrapped LLM
        const value = Reflect.get(target.llm, prop);
        if (typeof value === 'function') {
          return value.bind(target.llm);
        }
        return value;
      }
    });
  }
}

/**
 * Models known to be slower or need extended timeouts for complex tasks.
 * These models get extended timeout configuration to prevent premature failures
 * on multi-site comparison tasks.
 * 
 * Issue: https://github.com/VibeTechnologies/VibeWebAgent/issues/256
 */
const SLOW_MODELS = new Set([
  'gpt-5-mini',       // Smaller model, needs more time for complex reasoning
  'gpt-5-nano',       // Even smaller, needs extended timeouts
  'claude-3-haiku',   // Fast but may need extra time on complex tasks
]);

/**
 * Extended timeout configuration for slow models (in milliseconds).
 * Default LLMWithRetry timeout is 120s. Slow models get 180s.
 */
const SLOW_MODEL_TIMEOUT_CONFIG = {
  maxOverallTimeout: 180000,  // 180s total (vs 120s default)
  timeoutPerTry: 60000,       // 60s per attempt (vs 30s default)
};

/**
 * Check if a model is known to be slower and needs extended timeouts.
 * @param {string} model - Model name (without provider prefix)
 * @returns {boolean}
 */
function isSlowModel(model) {
  const normalizedModel = model.toLowerCase();
  return SLOW_MODELS.has(normalizedModel) || 
         Array.from(SLOW_MODELS).some(slow => normalizedModel.includes(slow));
}

// GitHub Copilot constants (following OpenCode pattern)
const COPILOT_HEADERS = {
  "User-Agent": "GitHubCopilotChat/0.32.4",
  "Editor-Version": "vscode/1.105.1",
  "Editor-Plugin-Version": "copilot-chat/0.32.4",
  "Copilot-Integration-Id": "vscode-chat",
};
const COPILOT_TOKEN_URL = "https://api.github.com/copilot_internal/v2/token";
const COPILOT_AUTH_STORAGE_KEY = "vibe_github_copilot_oauth";

// Claude Code (Anthropic OAuth) constants
const CLAUDE_CODE_AUTH_STORAGE_KEY = "vibe_claude_code_oauth";
const CLAUDE_CODE_CLIENT_ID = "9d1c250a-e61b-44d9-88ed-5944d1962f5e";
const CLAUDE_CODE_BETA_HEADERS = [
  "oauth-2025-04-20",
  "claude-code-20250219",
  "interleaved-thinking-2025-05-14",
  "fine-grained-tool-streaming-2025-05-14",
];

/**
 * Create a custom fetch function for GitHub Copilot that handles token refresh
 * Following OpenCode's pattern: refresh token on each request if expired
 */
function createCopilotFetch() {
  console.log("[Copilot] createCopilotFetch called - creating custom fetch function");
  return async (input, init) => {
    console.log("[Copilot] Custom fetch invoked for:", typeof input === 'string' ? input : input?.url);
    // Get stored auth from chrome.storage
    const result = await chrome.storage.local.get(COPILOT_AUTH_STORAGE_KEY);
    const storedValue = result[COPILOT_AUTH_STORAGE_KEY];
    if (!storedValue) {
      throw new Error("Not logged in to GitHub Copilot. Please connect in Settings.");
    }

    let auth;
    try {
      auth = JSON.parse(storedValue);
    } catch {
      throw new Error("Invalid GitHub Copilot auth data. Please reconnect in Settings.");
    }

    if (!auth.refresh) {
      throw new Error("GitHub Copilot refresh token missing. Please reconnect in Settings.");
    }

    // Refresh token if expired or about to expire (following OpenCode pattern)
    if (!auth.access || auth.expires < Date.now()) {
      console.log("[Copilot] Token expired, refreshing...");
      const response = await fetch(COPILOT_TOKEN_URL, {
        headers: {
          Accept: "application/json",
          Authorization: `Bearer ${auth.refresh}`,
          ...COPILOT_HEADERS,
        },
      });

      if (!response.ok) {
        if (response.status === 401) {
          // Clear invalid auth
          await chrome.storage.local.remove([COPILOT_AUTH_STORAGE_KEY, 'vibeApiKey_github-copilot']);
          throw new Error("GitHub Copilot session expired. Please reconnect in Settings.");
        }
        throw new Error(`Copilot token refresh failed: ${response.status}`);
      }

      const tokenData = await response.json();
      // Trim whitespace from token to avoid "invalid whitespace" errors
      auth.access = (tokenData.token || '').trim();
      auth.expires = tokenData.expires_at * 1000;

      // Save updated auth
      await chrome.storage.local.set({
        [COPILOT_AUTH_STORAGE_KEY]: JSON.stringify(auth),
        'vibeApiKey_github-copilot': auth.access
      });
      console.log("[Copilot] Token refreshed successfully");
    }

    // Detect if this is an agent call or vision request (following OpenCode pattern)
    let isAgentCall = false;
    let isVisionRequest = false;
    try {
      const body = typeof init?.body === "string" ? JSON.parse(init.body) : init?.body;
      if (body?.messages) {
        isAgentCall = body.messages.some(
          (msg) => msg.role && ["tool", "assistant"].includes(msg.role)
        );
        isVisionRequest = body.messages.some(
          (msg) =>
            Array.isArray(msg.content) &&
            msg.content.some((part) => part.type === "image_url")
        );
      }
    } catch {
      // Ignore parse errors
    }

    // Build headers (following OpenCode pattern)
    const headers = {
      ...(init?.headers || {}),
      ...COPILOT_HEADERS,
      Authorization: `Bearer ${auth.access}`,
      "Openai-Intent": "conversation-edits",
      "X-Initiator": isAgentCall ? "agent" : "user",
    };

    if (isVisionRequest) {
      headers["Copilot-Vision-Request"] = "true";
    }

    // Remove any conflicting auth headers (OpenAI SDK may set lowercase 'authorization')
    delete headers["x-api-key"];
    delete headers["authorization"];

    return fetch(input, {
      ...init,
      headers,
    });
  };
}

/**
 * Create a custom fetch function for Claude Code (Anthropic OAuth)
 * Handles token refresh and proper headers for Claude Pro/Max subscriptions
 */
function createClaudeCodeFetch() {
  return async (input, init) => {
    // Get stored auth from chrome.storage
    const result = await chrome.storage.local.get(CLAUDE_CODE_AUTH_STORAGE_KEY);
    const storedValue = result[CLAUDE_CODE_AUTH_STORAGE_KEY];
    if (!storedValue) {
      throw new Error("Not logged in to Claude. Please connect in Settings.");
    }

    let auth;
    try {
      auth = JSON.parse(storedValue);
    } catch {
      throw new Error("Invalid Claude auth data. Please reconnect in Settings.");
    }

    if (!auth.refresh) {
      throw new Error("Claude refresh token missing. Please reconnect in Settings.");
    }

    // Refresh token if expired
    if (!auth.access || auth.expires < Date.now()) {
      console.log("[ClaudeCode] Token expired, refreshing...");
      const response = await fetch("https://console.anthropic.com/v1/oauth/token", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          grant_type: "refresh_token",
          refresh_token: auth.refresh,
          client_id: CLAUDE_CODE_CLIENT_ID,
        }),
      });

      if (!response.ok) {
        if (response.status === 401) {
          // Clear invalid auth
          await chrome.storage.local.remove([CLAUDE_CODE_AUTH_STORAGE_KEY, 'vibeApiKey_claude-code']);
          throw new Error("Claude session expired. Please reconnect in Settings.");
        }
        throw new Error(`Claude token refresh failed: ${response.status}`);
      }

      const json = await response.json();
      auth.refresh = json.refresh_token;
      auth.access = json.access_token;
      auth.expires = Date.now() + json.expires_in * 1000;

      // Save updated auth
      await chrome.storage.local.set({
        [CLAUDE_CODE_AUTH_STORAGE_KEY]: JSON.stringify(auth),
        'vibeApiKey_claude-code': auth.access
      });
      console.log("[ClaudeCode] Token refreshed successfully");
    }

    // Merge beta headers
    const existingBeta = init?.headers?.["anthropic-beta"] || "";
    const existingBetas = existingBeta.split(",").map(b => b.trim()).filter(Boolean);
    const mergedBetas = [...new Set([...CLAUDE_CODE_BETA_HEADERS, ...existingBetas])].join(",");

    // Build headers
    const headers = {
      ...(init?.headers || {}),
      authorization: `Bearer ${auth.access}`,
      "anthropic-beta": mergedBetas,
    };

    // Remove conflicting headers
    delete headers["x-api-key"];

    return fetch(input, {
      ...init,
      headers,
    });
  };
}

export function getProviderModel(configuredModel) {
  const [provider, ...modelParts] = configuredModel.split(":");
  const model = modelParts.join(":");
  return { provider, model };
}
/**
 * Initialize a chat model based on the provider and model name
 * @param {string} modelString - Model string in format "provider:model" or just "model"
 * @param {Object} config - Configuration object with apiKey and other settings
 *          useResponsesApi: true, 
            outputVersion: "responses/v1"
 * @returns {Promise<Object>} - Returns the initialized chat model
 */
export function initChatModel(modelString, config = {}) {
  const { provider, model } = getProviderModel(modelString);

  // OpenAI-compatible provider base URLs
  // These providers work with ChatOpenAI by setting baseURL
  // VIBE_API_URL is set at build time: dev=https://api-dev.vibebrowser.app, prod=https://api.vibebrowser.app
  const VIBE_API_URL = process.env.VIBE_API_URL || 'https://api.vibebrowser.app';
  const baseURLs = {
    gemini: "https://generativelanguage.googleapis.com/v1beta",
    anthropic: "https://api.anthropic.com",
    openrouter: 'https://openrouter.ai/api/v1',
    vibe: `${VIBE_API_URL}/v1`,
    'vibe-tee': 'https://tee.vibebrowser.app/v1', // TEE backend with Intel TDX attestation (via Cloudflare Tunnel)
    'github-copilot': 'https://api.githubcopilot.com',
    // Additional OpenAI-compatible providers
    groq: 'https://api.groq.com/openai/v1',
    together: 'https://api.together.xyz/v1',
    deepinfra: 'https://api.deepinfra.com/v1/openai',
    fireworks: 'https://api.fireworks.ai/inference/v1',
    mistral: 'https://api.mistral.ai/v1',
    cerebras: 'https://api.cerebras.ai/v1',
    perplexity: 'https://api.perplexity.ai',
    xai: 'https://api.x.ai/v1',
    deepseek: 'https://api.deepseek.com/v1',
    ollama: 'http://localhost:11434/v1',
    'ollama-cloud': 'https://ollama.ai/v1',
    moonshot: 'https://api.moonshot.cn/v1',
    sambanova: 'https://api.sambanova.ai/v1',
    alibaba: 'https://dashscope.aliyuncs.com/compatible-mode/v1'
  };

  // Check if baseURL is provided in the configuration object
  // If not, use the default for the provider
  if (!config?.configuration?.baseURL) {
    if (!config.configuration) {
      config.configuration = {};
    }

    // First check if baseURL is at the top level of config
    if (config.baseURL) {
      config.configuration.baseURL = config.baseURL;
    }
    // Otherwise use the default URL for the provider if available
    else if (baseURLs[provider]) {
      config.configuration.baseURL = baseURLs[provider];
    }
  }

  // Get Langfuse callback handler if available
  const langfuseCallback = getLangfuseCallbackHandler();
  const callbacks = langfuseCallback ? [langfuseCallback] : [];

  switch (provider) {
    case 'google':
    case 'gemini':
    case 'google_genai':
      const geminiLLM = new ChatGoogleGenerativeAI({
        ...config,
        model,
        callbacks
      });
      return new LLMWithRetry(geminiLLM);

    case 'azure':
      // Azure OpenAI: model format is azure:deployment-name
      // Requires: baseURL (endpoint), apiKey
      const azureEndpoint = config.configuration?.baseURL || config.baseURL;
      if (!azureEndpoint) {
        throw new Error('Azure OpenAI requires baseURL (e.g., https://your-resource.openai.azure.com)');
      }
      if (!config.apiKey) {
        throw new Error('Azure OpenAI requires API key');
      }

      // Support both endpoint formats:
      // 1. Custom subdomain: https://my-resource.openai.azure.com
      // 2. Regional endpoint: https://eastus.api.cognitive.microsoft.com
      const customSubdomainMatch = azureEndpoint.match(/https?:\/\/([^.]+)\.openai\.azure\.com/);

      let azureLLMConfig = {
        azureOpenAIApiKey: config.apiKey,
        azureOpenAIApiDeploymentName: model,
        azureOpenAIApiVersion: '2024-08-01-preview',
        temperature: config.temperature,
        maxTokens: config.maxTokens,
        callbacks,
        ...(config.timeout !== undefined ? { timeout: config.timeout } : {}),
        ...(config.maxRetries !== undefined ? { maxRetries: config.maxRetries } : {})
      };

      if (customSubdomainMatch) {
        // Custom subdomain format - use instance name
        azureLLMConfig.azureOpenAIApiInstanceName = customSubdomainMatch[1];
      } else {
        // Regional endpoint format - use endpoint directly
        azureLLMConfig.azureOpenAIEndpoint = azureEndpoint;
      }

      const azureLLM = new AzureChatOpenAI(azureLLMConfig);
      return new LLMWithRetry(azureLLM);

    case 'deepseek':
      // DeepSeek: Requires special handling for tool calling
      // Known issue: DeepSeek tends to repeat tool calls in infinite loops when using ChatOpenAI
      // Fix: Disable parallel_tool_calls via DeepSeekLLMWrapper
      // Reference: https://github.com/langchain-ai/langchainjs/issues/7815
      const deepseekBaseLLM = new ChatOpenAI({
        ...config,
        model,
        callbacks,
        configuration: {
          baseURL: baseURLs.deepseek,
          ...config.configuration
        },
        ...(config.timeout !== undefined ? { timeout: config.timeout } : {}),
        ...(config.maxRetries !== undefined ? { maxRetries: config.maxRetries } : {})
      });
      // Wrap with DeepSeekLLMWrapper to ensure parallel_tool_calls: false on bindTools
      const deepseekWrappedLLM = new DeepSeekLLMWrapper(deepseekBaseLLM);
      return new LLMWithRetry(deepseekWrappedLLM);

    case 'vibe':
      // Vibe API: OpenAI-compatible endpoint via LiteLLM
      // Model format: vibe:gpt-5-mini, vibe:claude-3-sonnet, vibe:deepseek-v3.2, etc.
      const vibeBaseLLM = new ChatOpenAI({
        ...config,
        model,
        callbacks,
        configuration: {
          baseURL: baseURLs.vibe,
          ...config.configuration
        },
        ...(config.timeout !== undefined ? { timeout: config.timeout } : {}),
        ...(config.maxRetries !== undefined ? { maxRetries: config.maxRetries } : {})
      });
      
      // Check if this is a DeepSeek model routed through Vibe API
      // DeepSeek models need parallel_tool_calls: false to avoid infinite loops
      const isVibeDeepSeek = model.toLowerCase().includes('deepseek');
      if (isVibeDeepSeek) {
        console.log('[Vibe] Detected DeepSeek model, applying DeepSeekLLMWrapper');
        const vibeDeepSeekWrapped = new DeepSeekLLMWrapper(vibeBaseLLM);
        return new LLMWithRetry(vibeDeepSeekWrapped);
      }
      
      // Check if this is a slow model that needs extended timeouts
      // Issue #256: gpt-5-mini times out on multi-site comparison tasks
      if (isSlowModel(model)) {
        console.log(`[Vibe] Detected slow model ${model}, applying extended timeout (${SLOW_MODEL_TIMEOUT_CONFIG.maxOverallTimeout/1000}s)`);
        return new LLMWithRetry(vibeBaseLLM, SLOW_MODEL_TIMEOUT_CONFIG);
      }
      return new LLMWithRetry(vibeBaseLLM);

    case 'vibe-tee':
      // Vibe TEE API: Self-hosted DeepSeek on Intel TDX Confidential VM
      // Model format: vibe-tee:deepseek-r1, vibe-tee:deepseek-r1-7b
      // Attestation: http://tee.vibebrowser.app:4001/v1/attestation
      const vibeTeeBaseLLM = new ChatOpenAI({
        ...config,
        model,
        callbacks,
        apiKey: 'sk-tee-deepseek-key', // TEE backend API key
        configuration: {
          baseURL: baseURLs['vibe-tee'],
          ...config.configuration
        },
        ...(config.timeout !== undefined ? { timeout: config.timeout } : {}),
        ...(config.maxRetries !== undefined ? { maxRetries: config.maxRetries } : {})
      });
      
      // All TEE models are DeepSeek, apply wrapper for tool call handling
      console.log('[Vibe-TEE] Detected DeepSeek model on TEE, applying DeepSeekLLMWrapper');
      const vibeTeeWrapped = new DeepSeekLLMWrapper(vibeTeeBaseLLM);
      return new LLMWithRetry(vibeTeeWrapped);

    case 'chrome':
      // Chrome Built-in AI (Gemini Nano) - runs locally, no API key needed
      // Model format: chrome:gemini-nano
      // Uses XML tool emulation since Nano doesn't support native function calling
      const nanoLLM = new ChromeNanoLLM({
        temperature: config.temperature,
        topK: config.topK
      });
      return nanoLLM;  // No retry wrapper needed for local model

    case 'github-copilot':
      // GitHub Copilot: OpenAI-compatible endpoint with custom auth
      // Model format: github-copilot:gpt-4o, github-copilot:claude-3.5-sonnet
      // Requires OAuth authentication through GitHubCopilotService
      // Uses custom fetch that refreshes token on each request (following OpenCode pattern)
      const copilotLLM = new ChatOpenAI({
        ...config,
        model,
        callbacks,
        apiKey: "copilot", // Placeholder - actual auth is in custom fetch
        configuration: {
          ...config.configuration,
          baseURL: baseURLs['github-copilot'],
          fetch: createCopilotFetch(),  // Must be LAST to not be overwritten
        },
        ...(config.timeout !== undefined ? { timeout: config.timeout } : {}),
        ...(config.maxRetries !== undefined ? { maxRetries: config.maxRetries } : {})
      });
      return new LLMWithRetry(copilotLLM);

    case 'claude-code':
      // Claude Code: Anthropic API with OAuth authentication
      // Model format: claude-code:claude-sonnet-4-20250514
      // Uses Claude Pro/Max subscription via OAuth (no API key needed)
      const claudeCodeLLM = new ChatAnthropic({
        ...config,
        model,
        callbacks,
        anthropicApiKey: "oauth", // Placeholder - actual auth is in custom fetch
        clientOptions: {
          fetch: createClaudeCodeFetch(),
        },
        ...(config.timeout !== undefined ? { timeout: config.timeout } : {}),
        ...(config.maxRetries !== undefined ? { maxRetries: config.maxRetries } : {})
      });
      return new LLMWithRetry(claudeCodeLLM);

    default:
      // ChatOpenAI can accept baseURL in the configuration object
      const chatModelConfig = {
        ...config,
        model,
        callbacks,
        // Only set timeout if explicitly provided
        ...(config.timeout !== undefined ? { timeout: config.timeout } : {}),
        // Only set maxRetries if explicitly provided
        ...(config.maxRetries !== undefined ? { maxRetries: config.maxRetries } : {})
      };

      if (config.configuration?.baseURL) {
        chatModelConfig.baseUrl = config.configuration.baseURL;
      } else if (config.baseURL) {
        chatModelConfig.baseUrl = config.baseURL;
      }
      // https://v03.api.js.langchain.com/classes/_langchain_openai.ChatOpenAI.html
      const llm = new ChatOpenAI(chatModelConfig);
      
      // Check if this is a slow model that needs extended timeouts
      // Issue #256: gpt-5-mini times out on multi-site comparison tasks
      if (isSlowModel(model)) {
        console.log(`[LLM] Detected slow model ${model}, applying extended timeout (${SLOW_MODEL_TIMEOUT_CONFIG.maxOverallTimeout/1000}s)`);
        return new LLMWithRetry(llm, SLOW_MODEL_TIMEOUT_CONFIG);
      }
      // Wrap with retry functionality
      return new LLMWithRetry(llm);
  }
}
