/**
 * Image Compression Utility
 *
 * Compresses base64-encoded images to reduce token usage when sending to LLM vision APIs.
 *
 * Based on OpenAI Vision API best practices:
 * - Use "low" detail mode for simple tasks (85 tokens fixed)
 * - Resize images to optimal dimensions (max 1024-1280px)
 * - Use JPEG at 70-80% quality for best size/quality tradeoff
 * - Avoid sending 4K images when smaller versions work
 */

/**
 * Compress a base64 image data URL
 *
 * @param {string} dataUrl - Base64 data URL (e.g., "data:image/png;base64,...")
 * @param {Object} options - Compression options
 * @param {number} options.maxWidth - Maximum width in pixels (default: 1024)
 * @param {number} options.maxHeight - Maximum height in pixels (default: 1024)
 * @param {number} options.quality - JPEG quality 0-100 (default: 75)
 * @param {string} options.format - Output format "jpeg" or "png" (default: "jpeg")
 * @returns {Promise<string>} Compressed base64 data URL
 */
export async function compressImage(dataUrl, options = {}) {
  // DISABLED: Image compression temporarily disabled due to service worker compatibility issues
  // Return original dataUrl without processing
  console.log('📸 Image compression disabled, using original image');
  return Promise.resolve(dataUrl);
}

/**
 * Get optimal compression settings based on use case
 *
 * @param {string} detail - Detail level: "low" | "high" | "auto"
 * @returns {Object} Compression options
 */
export function getOptimalSettings(detail = 'auto') {
  switch (detail) {
    case 'low':
      // Optimized for minimal token usage
      // Good for: Simple detection, presence/absence checks, basic OCR
      return {
        maxWidth: 512,
        maxHeight: 512,
        quality: 70,
        format: 'jpeg'
      };

    case 'high':
      // Balanced quality and token usage
      // Good for: Detailed analysis, charts, complex layouts
      return {
        maxWidth: 1280,
        maxHeight: 1280,
        quality: 80,
        format: 'jpeg'
      };

    case 'auto':
    default:
      // Default balanced settings
      return {
        maxWidth: 1024,
        maxHeight: 1024,
        quality: 75,
        format: 'jpeg'
      };
  }
}

/**
 * Estimate token usage for an image
 * Based on OpenAI's token calculation:
 * - Low detail: fixed 85 tokens
 * - High detail: calculated based on 512px tiles
 *
 * @param {number} width - Image width in pixels
 * @param {number} height - Image height in pixels
 * @param {string} detail - Detail level: "low" | "high"
 * @returns {number} Estimated token count
 */
export function estimateImageTokens(width, height, detail = 'high') {
  if (detail === 'low') {
    return 85; // Fixed cost for low detail
  }

  // High detail calculation:
  // 1. Resize to fit within 2048x2048
  let scaledWidth = width;
  let scaledHeight = height;

  if (width > 2048 || height > 2048) {
    const scale = Math.min(2048 / width, 2048 / height);
    scaledWidth = Math.round(width * scale);
    scaledHeight = Math.round(height * scale);
  }

  // 2. Scale shortest side to 768px
  const scale = 768 / Math.min(scaledWidth, scaledHeight);
  scaledWidth = Math.round(scaledWidth * scale);
  scaledHeight = Math.round(scaledHeight * scale);

  // 3. Count 512px tiles
  const tilesWidth = Math.ceil(scaledWidth / 512);
  const tilesHeight = Math.ceil(scaledHeight / 512);
  const tiles = tilesWidth * tilesHeight;

  // Each tile costs 170 tokens, plus 85 base tokens
  return tiles * 170 + 85;
}
