/**
 * HTTP Request Interceptor for debugging LLM API calls
 * Captures and logs the actual JSON payloads sent to API endpoints
 */

export function setupHttpInterceptor(verbose = false) {
  if (!verbose) {
    // Disable interceptor if verbose is false
    if (globalThis._originalFetch) {
      globalThis.fetch = globalThis._originalFetch;
      console.log('🔍 HTTP Interceptor disabled');
    }
    return;
  }

  console.log('🔍 HTTP Interceptor enabled - will log all LLM API calls');

  // Store original fetch ONCE
  if (!globalThis._originalFetch) {
    globalThis._originalFetch = globalThis.fetch;
  }
  const originalFetch = globalThis._originalFetch;
  
  // Track if we've already logged tools
  let toolsAlreadyLogged = false;
  
  // Override fetch to log requests
  globalThis.fetch = async function(...args) {
    const [url, options = {}] = args;

    // Log all fetch requests when verbose is enabled for debugging
    console.log(`[HTTP INTERCEPTOR] Fetch call to: ${typeof url === 'string' ? url.substring(0, 100) : url}`);

    // Check if this is an OpenAI API call
    if (url && (url.includes('openrouter') || url.includes('openai') || url.includes('api.anthropic.com'))) {      
      let requestPayload = null;
      
      // Parse request body
      if (options.body) {
        try {
          if (typeof options.body === 'string') {
            requestPayload = JSON.parse(options.body);
          } else if (options.body instanceof ReadableStream) {
            requestPayload = '[ReadableStream]';
          } else {
            requestPayload = options.body;
          }
        } catch (e) {
          requestPayload = '[Parse Error]';
        }
      }
      
      // Make the actual request
      const startTime = Date.now();
      try {
        const response = await originalFetch.apply(this, args);
        const duration = Date.now() - startTime;
        const clonedResponse = response.clone();
        const responseJson = await clonedResponse.json();

        console.log(`${options.method || 'POST'} ${url} ${JSON.stringify(requestPayload)}: ${response.status} ${JSON.stringify(responseJson)}`);
        
        return response;
      } catch (error) {
        const duration = Date.now() - startTime;
        console.log(`${options.method || 'POST'} ${url} ${JSON.stringify(requestPayload)}: ERROR ${error.message}`);
        throw error;
      }
    }
    
    // For non-OpenAI requests, just pass through
    return originalFetch.apply(this, args);
  };
  
}

export function disableHttpInterceptor() {
  // Restore original fetch if it was stored
  if (globalThis.originalFetch) {
    globalThis.fetch = globalThis.originalFetch;
    console.log('HTTP Interceptor disabled');
  }
}

export function resetHttpInterceptor() {
  // Reset the tools logged flag for new sessions
  if (globalThis.fetch && globalThis.fetch.toolsAlreadyLogged !== undefined) {
    globalThis.fetch.toolsAlreadyLogged = false;
  }
}
