/**
 * LLM Wrapper with Retry Logic
 * Wraps any LangChain LLM with exponential backoff retry functionality
 */

export class LLMWithRetry {
  constructor(llm, retryConfig = {}) {
    this.llm = llm;
    this.retryConfig = {
      maxRetries: 3,
      baseDelay: 1000, // 1 second
      maxDelay: 30000, // 30 seconds
      maxOverallTimeout: 120000, // 120 seconds max per request (increased for complex subagent tasks)
      timeoutPerTry: 30000, // 30 seconds max per attempt
      jitter: true,
      ...retryConfig
    };

    // Preserve all original LLM properties and methods
    return new Proxy(this, {
      get(target, prop, receiver) {
        // If it's one of our custom methods, return it
        if (prop in target) {
          return Reflect.get(target, prop, receiver);
        }

        // For the invoke method, use our retry wrapper
        if (prop === 'invoke') {
          return target.invokeWithRetry.bind(target);
        }

        // For all other properties/methods, delegate to the wrapped LLM
        const value = Reflect.get(target.llm, prop);

        // If it's a function, bind it to the original LLM
        if (typeof value === 'function') {
          // For bindTools, wrap the result so it also has retry logic on invoke
          if (prop === 'bindTools') {
            return (...args) => {
              const boundLLM = value.apply(target.llm, args);
              return target.wrapBoundLLM(boundLLM);
            };
          }
          return value.bind(target.llm);
        }

        return value;
      }
    });
  }

  /**
   * Wrap a bound LLM (e.g., from bindTools) to maintain retry/timeout behavior
   * @param {Object} boundLLM - The bound LLM from bindTools
   * @returns {Object} Proxied LLM with retry behavior on invoke
   */
  wrapBoundLLM(boundLLM) {
    const self = this;
    return new Proxy(boundLLM, {
      get(target, prop, receiver) {
        if (prop === 'invoke') {
          // Wrap the invoke method with timeout (but not full retry, since chains may have different semantics)
          return async (args, options = {}) => {
            const timeoutMs = self.retryConfig.maxOverallTimeout;
            const startTime = Date.now();
            
            const timeoutPromise = new Promise((_, reject) => {
              setTimeout(() => {
                reject(new Error(`LLM chain request timeout after ${timeoutMs}ms`));
              }, timeoutMs);
            });

            try {
              const result = await Promise.race([
                target.invoke(args, options),
                timeoutPromise
              ]);
              console.log(`[LLM] Chain invoke completed in ${Date.now() - startTime}ms`);
              return result;
            } catch (error) {
              console.error(`[LLM] Chain invoke failed after ${Date.now() - startTime}ms:`, error.message);
              throw error;
            }
          };
        }
        return Reflect.get(target, prop, receiver);
      }
    });
  }

  /**
   * Invoke with retry functionality
   * @param {any} args - Arguments to pass to the LLM invoke method
   * @param {Object} options - Additional options (including timeout)
   * @param {number} attempt - Current attempt number (1-indexed)
   * @param {number} startTime - Start time for overall timeout tracking
   * @returns {Promise<any>} LLM response
   */
  async invokeWithRetry(args, options = {}, attempt = 1, startTime = Date.now()) {
    try {
      console.log(`[LLM RETRY] Attempt ${attempt}/${this.retryConfig.maxRetries + 1}`);

      // Calculate remaining time and determine timeout for this attempt
      const elapsed = Date.now() - startTime;
      const remainingTime = this.retryConfig.maxOverallTimeout - elapsed;

      if (remainingTime <= 0) {
        console.error(`[LLM RETRY] Overall timeout exceeded before new attempt.`);
        throw new Error(`LLM retry timeout exceeded: ${elapsed}ms elapsed.`);
      }

      const timeoutMs = Math.min(this.retryConfig.timeoutPerTry, remainingTime);

      // Create timeout promise to ensure request doesn't hang indefinitely
      const timeoutPromise = new Promise((_, reject) => {
        setTimeout(() => {
          reject(new Error(`LLM request timeout after ${timeoutMs}ms`));
        }, timeoutMs);
      });

      // Race between the actual request and timeout
      return await Promise.race([
        this.llm.invoke(args, options),
        timeoutPromise
      ]);
    } catch (error) {
      // Enhanced error logging with full error details
      console.error(`[LLM RETRY] Attempt ${attempt} failed:`, {
        message: error.message,
        name: error.name,
        code: error.code,
        status: error.status,
        stack: error.stack?.split('\n').slice(0, 3).join('\n'), // First 3 lines of stack
        fullError: JSON.stringify(error, Object.getOwnPropertyNames(error))
      });

      // Check overall timeout (prevent infinite retry loops)
      const elapsed = Date.now() - startTime;
      if (elapsed > this.retryConfig.maxOverallTimeout) {
        console.error(`[LLM RETRY] Overall timeout exceeded (${elapsed}ms > ${this.retryConfig.maxOverallTimeout}ms)`);
        throw new Error(`LLM retry timeout exceeded: ${elapsed}ms elapsed, original error: ${error.message}`);
      }

      // Check if we should retry
      const shouldRetry = attempt <= this.retryConfig.maxRetries && this.shouldRetry(error);
      if (shouldRetry) {
        const delay = this.calculateBackoffDelay(attempt);
        const retryReason = this.getRetryReason(error);
        console.log(`[LLM RETRY] Retrying in ${delay}ms due to: ${retryReason}`);

        // Wait before retrying
        await new Promise(resolve => setTimeout(resolve, delay));

        return this.invokeWithRetry(args, options, attempt + 1, startTime);
      }

      // Max retries exceeded or non-retryable error
      console.error(`[LLM RETRY] Max retries exceeded or non-retryable error. Final attempt: ${attempt}, elapsed: ${elapsed}ms`);
      throw error;
    }
  }

  /**
   * Determine if an error is retryable
   * @param {Error} error - The error to check
   * @returns {boolean} Whether the error should trigger a retry
   */
  shouldRetry(error) {
    const message = error.message?.toLowerCase() || '';
    const errorCode = error.code?.toLowerCase() || '';
    const statusCode = error.status?.toString() || '';

    // Retry on rate limiting and temporary network issues
    const retryableErrors = [
      'rate limit',
      'too many requests',
      'timeout',
      'connection',
      'network',
      'econnreset',
      'enotfound',
      'econnrefused',
      'socket hang up',
      'request timeout',
      'fetch failed',
      'abort',
      'cancelled',
      'llm request timeout'
    ];

    const retryableStatusCodes = ['429', '500', '502', '503', '504'];
    const retryableErrorCodes = ['econnreset', 'enotfound', 'econnrefused', 'etimedout'];

    // Check message content
    const hasRetryableMessage = retryableErrors.some(keyword => message.includes(keyword));

    // Check status codes
    const hasRetryableStatus = retryableStatusCodes.some(code =>
      statusCode.includes(code) || message.includes(code)
    );

    // Check error codes
    const hasRetryableErrorCode = retryableErrorCodes.some(code =>
      errorCode.includes(code) || message.includes(code)
    );

    return hasRetryableMessage || hasRetryableStatus || hasRetryableErrorCode;
  }

  /**
   * Get a human-readable reason for why we're retrying
   * @param {Error} error - The error that triggered the retry
   * @returns {string} Human-readable retry reason
   */
  getRetryReason(error) {
    const message = error.message?.toLowerCase() || '';
    const statusCode = error.status?.toString() || '';

    if (message.includes('rate limit') || message.includes('429') || statusCode === '429') {
      return 'Rate limit exceeded';
    }
    if (message.includes('timeout') || message.includes('etimedout')) {
      return 'Request timeout';
    }
    if (message.includes('network') || message.includes('connection') || message.includes('econnreset')) {
      return 'Network connectivity issue';
    }
    if (message.includes('500') || statusCode === '500') {
      return 'Server internal error';
    }
    if (message.includes('502') || statusCode === '502') {
      return 'Bad gateway';
    }
    if (message.includes('503') || statusCode === '503') {
      return 'Service unavailable';
    }
    if (message.includes('504') || statusCode === '504') {
      return 'Gateway timeout';
    }
    if (message.includes('fetch failed') || message.includes('abort')) {
      return 'Request aborted or failed';
    }

    return `Temporary error: ${error.message.substring(0, 100)}`;
  }

  /**
   * Calculate exponential backoff delay with jitter
   * @param {number} attempt - Current attempt number (1-indexed)
   * @returns {number} Delay in milliseconds
   */
  calculateBackoffDelay(attempt) {
    // Exponential backoff: baseDelay * 2^(attempt-1)
    let delay = this.retryConfig.baseDelay * Math.pow(2, attempt - 1);

    // Cap at maxDelay
    delay = Math.min(delay, this.retryConfig.maxDelay);

    // Add jitter to prevent thundering herd
    if (this.retryConfig.jitter) {
      const jitterRange = delay * 0.1; // ±10% jitter
      const jitter = (Math.random() - 0.5) * 2 * jitterRange;
      delay += jitter;
    }

    return Math.max(delay, 0); // Ensure non-negative
  }
}