/**
 * GetPageContentTool - Auto-called tool for page content extraction
 * 
 * This tool is NOT called by the LLM but automatically invoked by the agent
 * to provide page context in messages. The agent doesn't know or care about
 * the underlying implementation (Playwright, Extension, etc.)
 */

export class GetPageContentTool {
  constructor(contentProvider) {
    if (!contentProvider) {
      throw new Error('GetPageContentTool requires a content provider');
    }
    this.contentProvider = contentProvider;
  }

  /**
   * Get the current page content
   * This is called automatically by the agent, not by LLM decision
   * @param {string} [targetId] - Optional target identifier
   * @returns {Promise<{content: string, htmlContent: string, url: string, title: string, tabId?: number}>}
   */
  async execute(targetId) {
    try {
      const result = await this.contentProvider.getPageContent(targetId);

      // Ensure we always return the expected structure
      return {
        content: result.content || 'Unable to extract page content',
        htmlContent: result.htmlContent || '',
        url: result.url || '',
        title: result.title || '',
        tabId: result.tabId // Include tab ID if provided by the content provider
      };
    } catch (error) {
      console.error('[GetPageContentTool] Error:', error);
      // Return a valid structure even on error
      return {
        content: `Error extracting page content: ${error.message}`,
        htmlContent: '',
        url: '',
        title: '',
        tabId: undefined
      };
    }
  }

  /**
   * Check if content looks like it needs authentication or has issues
   */
  validateContent(content) {
    const issues = [];
    
    if (!content || content.length < 100) {
      issues.push('Content seems too short');
    }
    
    if (content.includes('Sign in') || content.includes('Log in')) {
      issues.push('Page may require authentication');
    }
    
    if (content.includes('Access denied') || content.includes('403')) {
      issues.push('Access denied to page');
    }
    
    return {
      isValid: issues.length === 0,
      issues
    };
  }
}