import { jest } from "@jest/globals";

// Mock Chrome APIs for testing
const chrome = {
  tabs: {
    update: jest.fn(),
    create: jest.fn(),
    get: jest.fn(),
    query: jest.fn(),
    remove: jest.fn(),
    reload: jest.fn(),
    captureVisibleTab: jest.fn(),
    sendMessage: jest.fn(),
  },
  scripting: {
    executeScript: jest.fn(),
  },
  runtime: {
    sendMessage: jest.fn(),
    onMessage: {
      addListener: jest.fn(),
    },
  },
  webNavigation: {
    onBeforeNavigate: {
      addListener: jest.fn(),
    },
  },
  sidePanel: {
    open: jest.fn(),
    setPanelBehavior: jest.fn(),
  },
};

// Mock the vibe mojo interface
const mockVibeMojoInterface = {
  initialize: jest.fn().mockResolvedValue(true),
  executeBrowserAction: jest.fn(),
  getPageContext: jest.fn(),
  getAvailableTools: jest.fn(),
  setApiKey: jest.fn(),
  hasApiKey: jest.fn(),
  validateApiKey: jest.fn(),
  isInitialized: false,
};

// Mock LangChain modules
const mockLLM = {
  bindTools: jest.fn(),
  invoke: jest.fn(),
};

const mockStateGraph = {
  addNode: jest.fn().mockReturnThis(),
  addEdge: jest.fn().mockReturnThis(),
  addConditionalEdges: jest.fn().mockReturnThis(),
  compile: jest.fn().mockReturnValue({
    invoke: jest.fn(),
  }),
};

// Mock the tool function from @langchain/core/tools
const tool = jest.fn((func, options) => ({
  name: options.name,
  description: options.description,
  schema: options.schema,
  call: func
}));

// Mock the z object from zod
const z = {
  object: jest.fn((schema) => ({ schema })),
  string: jest.fn(() => ({
    describe: jest.fn().mockReturnThis(),
    optional: jest.fn().mockReturnThis(),
  })),
  number: jest.fn(() => ({
    describe: jest.fn().mockReturnThis(),
    optional: jest.fn().mockReturnThis(),
  })),
  array: jest.fn(() => ({
    describe: jest.fn().mockReturnThis(),
    optional: jest.fn().mockReturnThis(),
  })),
  boolean: jest.fn(() => ({
    describe: jest.fn().mockReturnThis(),
    optional: jest.fn().mockReturnThis(),
  })),
  enum: jest.fn(() => ({
    describe: jest.fn().mockReturnThis(),
    optional: jest.fn().mockReturnThis(),
  })),
};

function makeTool(name, description, func, schema) {
  return tool(func, {
    name,
    description,
    schema,
  });
}

export const browserTools = [
  makeTool(
    "navigate_to_url", 
    "Navigate the browser to a specific URL", 
    async ({ tabId, url }) => {
      // Use Chrome extension API for simple navigation
      await chrome.tabs.update(tabId, { url });
      return `Successfully navigated to ${url}`;
    },
    z.object({
      tabId: z.number().describe("The ID of the tab to navigate"),
      url: z.string().describe("The URL to navigate to")
    })
  ),
  makeTool(
    "fill_form_field", 
    "Fill a form field on the current page", 
    async ({ tabId, selector, value }) => {
      // Use Chrome extension API for simple form filling
      await chrome.tabs.sendMessage(tabId, { type: "fillForm", data: { selector, value } });
      return `Filled field ${selector} with "${value}"`;
    },
    z.object({
      tabId: z.number().describe("The ID of the tab"),
      selector: z.string().describe("CSS selector for the form field"),
      value: z.string().describe("Value to fill in the field")
    })
  ),
  makeTool(
    "click_element", 
    "Click an element on the current page", 
    async ({ tabId, selector }) => {
      // Use Chrome extension API for simple clicking
      await chrome.tabs.sendMessage(tabId, { type: "clickElement", data: { selector } });
      return `Clicked element ${selector}`;
    },
    z.object({
      tabId: z.number().describe("The ID of the tab"),
      selector: z.string().describe("CSS selector for the element to click")
    })
  ),
  makeTool(
    "done", 
    "Mark the task as completed with a summary", 
    async ({ summary }) => {
      return `Task completed: ${summary}`;
    },
    z.object({
      summary: z.string().describe("A brief summary of what was accomplished")
    })
  ),
  makeTool(
    "show_user_options",
    "Present multiple options to the user for selection",
    async ({ message, options, context }) => {
      try {
        // Send options to side panel for user selection
        const response = await chrome.runtime.sendMessage({
          type: "showUserOptions",
          message: message,
          options: options,
          context: context || ""
        });
        
        if (response && response.success) {
          return `User selected: ${response.selectedOption}`;
        } else {
          return `Failed to get user selection: ${response?.error || "Unknown error"}`;
        }
      } catch (error) {
        return `Error showing options: ${error.message}`;
      }
    },
    z.object({
      message: z.string().describe("Message to show with the options"),
      options: z.array(z.string()).describe("Array of options for the user to choose from"),
      context: z.string().optional().describe("Additional context about the choice")
    })
  ),
  makeTool(
    "request_user_confirmation",
    "Ask user to confirm an action before proceeding",
    async ({ action, details, price }) => {
      try {
        // Send confirmation request to side panel
        const response = await chrome.runtime.sendMessage({
          type: "requestConfirmation",
          action: action,
          details: details,
          price: price || null
        });
        
        if (response && response.success) {
          return response.confirmed ? "User confirmed the action" : "User declined the action";
        } else {
          return `Failed to get user confirmation: ${response?.error || "Unknown error"}`;
        }
      } catch (error) {
        return `Error requesting confirmation: ${error.message}`;
      }
    },
    z.object({
      action: z.string().describe("The action you want to confirm (e.g., 'Book flight')"),
      details: z.string().describe("Details about what will be done (e.g., 'United Airlines SFO->JFK')"),
      price: z.string().optional().describe("Price if applicable (e.g., '$299')")
    })
  ),
  makeTool("create_tab", "Create a new tab with optional URL", async ({ url, active = true }) => {
    const tab = await chrome.tabs.create({ url, active });
    return `Created new tab with id ${tab.id} and url ${tab.url}`;
  }),
  makeTool("switch_tab", "Switch to a specific tab by ID or URL pattern", async ({ tabId, urlPattern }) => {
    let targetTab;
    if (tabId) {
      targetTab = await chrome.tabs.get(tabId);
    } else if (urlPattern) {
      const tabs = await chrome.tabs.query({});
      targetTab = tabs.find(tab => tab.url && tab.url.includes(urlPattern));
    }
    if (targetTab) {
      await chrome.tabs.update(targetTab.id, { active: true });
      return `Switched to tab ${targetTab.id}`;
    }
    return "No matching tab found";
  }),
  makeTool(
    "get_page_dom", 
    "Get comprehensive DOM information including all interactive elements", 
    async ({ tabId }) => {
    try {
      // Try Mojo interface first for comprehensive context
      const context = await mockVibeMojoInterface.getPageContext();
      return context;
    } catch (error) {
      // Fall back to Chrome extension API
      const [result] = await chrome.scripting.executeScript({
        target: { tabId },
        func: () => {
          return {
            url: window.location.href,
            title: document.title,
            inputs: Array.from(document.querySelectorAll("input, select, textarea")).map(el => ({
              tag: el.tagName.toLowerCase(),
              type: el.type,
              id: el.id,
              name: el.name,
              className: el.className,
              placeholder: el.placeholder,
              value: el.value,
              required: el.required,
              disabled: el.disabled,
              visible: el.offsetWidth > 0 && el.offsetHeight > 0
            })),
            buttons: Array.from(document.querySelectorAll("button, input[type=\"submit\"], input[type=\"button\"], [role=\"button\"]")).map(el => ({
              tag: el.tagName.toLowerCase(),
              type: el.type,
              id: el.id,
              className: el.className,
              text: el.textContent?.trim(),
              disabled: el.disabled,
              visible: el.offsetWidth > 0 && el.offsetHeight > 0
            })),
            links: Array.from(document.querySelectorAll("a[href]")).map(el => ({
              href: el.href,
              text: el.textContent?.trim(),
              id: el.id,
              className: el.className,
              visible: el.offsetWidth > 0 && el.offsetHeight > 0
            })),
            forms: Array.from(document.querySelectorAll("form")).map(form => ({
              id: form.id,
              action: form.action,
              method: form.method,
              fieldCount: form.elements.length
            })),
            text_content: document.body.innerText?.substring(0, 2000) + "..."
          };
        }
      });
      return JSON.stringify(result.result);
    }
    },
    z.object({
      tabId: z.number().describe("The ID of the tab to analyze")
    })
  ),
  makeTool("find_elements", "Find elements by CSS selector or text content", async ({ tabId, selector, text }) => {
    // Use Chrome extension API for simple element finding
    const [result] = await chrome.scripting.executeScript({
      target: { tabId },
      func: (selector, text) => {
        let elements = [];
        if (selector) {
          elements = Array.from(document.querySelectorAll(selector));
        } else if (text) {
          const walker = document.createTreeWalker(document.body, NodeFilter.SHOW_TEXT, null, false);
          let node;
          while (node = walker.nextNode()) {
            if (node.textContent.toLowerCase().includes(text.toLowerCase())) {
              elements.push(node.parentElement);
            }
          }
        }
        return elements.map(el => ({
          tag: el.tagName?.toLowerCase(),
          id: el.id,
          className: el.className,
          text: el.textContent?.trim()?.substring(0, 100),
          visible: el.offsetWidth > 0 && el.offsetHeight > 0
        }));
      },
      args: [selector, text]
    });
    return JSON.stringify(result.result);
  }),
  makeTool("scroll_page", "Scroll the page in specified direction or to element", async ({ tabId, direction, selector, pixels }) => {
    await chrome.scripting.executeScript({
      target: { tabId },
      func: (direction, selector, pixels) => {
        if (selector) {
          const element = document.querySelector(selector);
          if (element) {
            element.scrollIntoView({ behavior: "smooth", block: "center" });
          }
        } else if (pixels) {
          window.scrollBy(0, pixels);
        } else {
          switch (direction) {
            case "up": window.scrollBy(0, -500); break;
            case "down": window.scrollBy(0, 500); break;
            case "top": window.scrollTo(0, 0); break;
            case "bottom": window.scrollTo(0, document.body.scrollHeight); break;
          }
        }
      },
      args: [direction, selector, pixels]
    });
    return `Scrolled page (${direction || selector || pixels})`;
  }),
  makeTool("type_text", "Type text character by character (simulates real typing)", async ({ tabId, selector, text, delay = 50 }) => {
    await chrome.scripting.executeScript({
      target: { tabId },
      func: async (sel, txt, delay) => {
        const input = document.querySelector(sel);
        if (input) {
          input.focus();
          input.value = "";
          for (let i = 0; i < txt.length; i++) {
            input.value += txt[i];
            input.dispatchEvent(new Event("input", { bubbles: true }));
            await new Promise(resolve => setTimeout(resolve, delay));
          }
          input.dispatchEvent(new Event("change", { bubbles: true }));
        }
      },
      args: [selector, text, delay]
    });
    return `Typed text into ${selector}`;
  }),
  makeTool("inject_javascript", "Execute custom JavaScript code on the page", async ({ tabId, code }) => {
    const [result] = await chrome.scripting.executeScript({
      target: { tabId },
      func: (jsCode) => {
        try {
          const result = eval(jsCode);
          return { success: true, result: result };
        } catch (error) {
          return { success: false, error: error.message };
        }
      },
      args: [code]
    });
    return JSON.stringify(result.result);
  }),
  makeTool("wait_for_element", "Wait for an element to appear on the page", async ({ tabId, selector, timeout = 10000 }) => {
    const [result] = await chrome.scripting.executeScript({
      target: { tabId },
      func: (sel, timeout) => {
        return new Promise((resolve) => {
          const startTime = Date.now();
          const checkElement = () => {
            const element = document.querySelector(sel);
            if (element && element.offsetWidth > 0 && element.offsetHeight > 0) {
              resolve({ success: true, found: true });
            } else if (Date.now() - startTime > timeout) {
              resolve({ success: false, found: false, error: "Timeout" });
            } else {
              setTimeout(checkElement, 100);
            }
          };
          checkElement();
        });
      },
      args: [selector, timeout]
    });
    return JSON.stringify(result.result);
  }),
  makeTool(
    "take_screenshot", 
    "Take a screenshot of the current page", 
    async () => {
      const dataUrl = await chrome.tabs.captureVisibleTab();
      return `Screenshot captured (truncated): ${dataUrl.substring(0, 100)}...`;
    },
    z.object({})
  ),
  makeTool("send_keys", "Send specific keyboard keys to an element", async ({ tabId, selector, keys }) => {
    await chrome.scripting.executeScript({
      target: { tabId },
      func: (sel, keySequence) => {
        const element = document.querySelector(sel);
        if (element) {
          element.focus();
          for (const key of keySequence) {
            const event = new KeyboardEvent("keydown", { key: key, bubbles: true });
            element.dispatchEvent(event);
            const eventUp = new KeyboardEvent("keyup", { key: key, bubbles: true });
            element.dispatchEvent(eventUp);
          }
        }
      },
      args: [selector, keys]
    });
    return `Sent keys ${keys.join(", ")} to ${selector}`;
  }),
  makeTool("hover_element", "Hover over an element", async ({ tabId, selector }) => {
    await chrome.scripting.executeScript({
      target: { tabId },
      func: (sel) => {
        const element = document.querySelector(sel);
        if (element) {
          const event = new MouseEvent("mouseover", { bubbles: true, cancelable: true });
          element.dispatchEvent(event);
        }
      },
      args: [selector]
    });
    return `Hovered over ${selector}`;
  }),
  makeTool("right_click", "Right-click on an element", async ({ tabId, selector }) => {
    await chrome.scripting.executeScript({
      target: { tabId },
      func: (sel) => {
        const element = document.querySelector(sel);
        if (element) {
          const event = new MouseEvent("contextmenu", { bubbles: true, cancelable: true });
          element.dispatchEvent(event);
        }
      },
      args: [selector]
    });
    return `Right-clicked ${selector}`;
  }),
  makeTool("get_element_attributes", "Get detailed attributes and styles of an element", async ({ tabId, selector }) => {
    const [result] = await chrome.scripting.executeScript({
      target: { tabId },
      func: (sel) => {
        const element = document.querySelector(sel);
        if (element) {
          const styles = window.getComputedStyle(element);
          return {
            tag: element.tagName.toLowerCase(),
            id: element.id,
            className: element.className,
            attributes: Object.fromEntries(
              Array.from(element.attributes).map(attr => [attr.name, attr.value])
            ),
            text: element.textContent?.trim(),
            innerHTML: element.innerHTML,
            outerHTML: element.outerHTML,
            computedStyles: {
              display: styles.display,
              visibility: styles.visibility,
              position: styles.position,
              top: styles.top,
              left: styles.left,
              width: styles.width,
              height: styles.height,
              color: styles.color,
              backgroundColor: styles.backgroundColor,
              fontSize: styles.fontSize,
              fontFamily: styles.fontFamily
            },
            boundingRect: element.getBoundingClientRect(),
            visible: element.offsetWidth > 0 && element.offsetHeight > 0
          };
        }
        return null;
      },
      args: [selector]
    });
    return JSON.stringify(result.result);
  }),
  makeTool("wait_for_condition", "Wait for a JavaScript condition to be true", async ({ tabId, condition, timeout = 10000 }) => {
    const [result] = await chrome.scripting.executeScript({
      target: { tabId },
      func: (cond, timeout) => {
        return new Promise((resolve) => {
          const startTime = Date.now();
          const checkCondition = () => {
            try {
              const result = eval(cond);
              if (result) {
                resolve({ success: true, result: result });
              } else if (Date.now() - startTime > timeout) {
                resolve({ success: false, found: false, error: "Timeout" });
              } else {
                setTimeout(checkCondition, 100);
              }
            } catch (error) {
              resolve({ success: false, error: error.message });
            }
          };
          checkCondition();
        });
      },
      args: [condition, timeout]
    });
    return JSON.stringify(result.result);
  }),
  makeTool("get_page_performance", "Get page performance metrics", async ({ tabId }) => {
    const [result] = await chrome.scripting.executeScript({
      target: { tabId },
      func: () => {
        const perfData = performance.getEntriesByType("navigation")[0];
        return {
          loadTime: perfData.loadEventEnd - perfData.loadEventStart,
          domContentLoaded: perfData.domContentLoadedEventEnd - perfData.domContentLoadedEventStart,
          firstPaint: performance.getEntriesByName("first-paint")[0]?.startTime || 0,
          firstContentfulPaint: performance.getEntriesByName("first-contentful-paint")[0]?.startTime || 0,
          pageSize: document.documentElement.outerHTML.length,
          resourceCount: performance.getEntriesByType("resource").length
        };
      }
    });
    return JSON.stringify(result.result);
  }),
  makeTool("close_tab", "Close a specific tab", async ({ tabId }) => {
    await chrome.tabs.remove(tabId);
    return `Closed tab ${tabId}`;
  }),
  makeTool("reload_page", "Reload the current page", async ({ tabId }) => {
    await chrome.tabs.reload(tabId);
    return `Reloaded page in tab ${tabId}`;
  }),
];
