// @ts-nocheck
/**
 * Vibe AI Browser - Personalized Suggestions Generator
 * Copyright (c) 2025 Vibe Technologies, LLC
 *
 * Generates personalized action suggestions based on user history using LLM
 */

import { initChatModel } from "./utils/initChatModel.js";
import { z } from "zod";

/**
 * Schema for structured suggestion output
 */
const SuggestionSchema = z.object({
  suggestions: z.array(
    z.object({
      emoji: z.string().describe("A single emoji that represents this action"),
      title: z.string().describe("Short, actionable title (5-8 words max)"),
      description: z.string().describe("Brief description of what this does (10-15 words)")
    })
  ).length(5).describe("Array of exactly 5 personalized suggestions")
});

/**
 * Generate personalized suggestions based on user history
 * @param {Object} config - LLM configuration (apiKey, model, baseURL, etc.)
 * @param {Array} recentSessions - Recent session data with queries and messages
 * @param {Array} bookmarks - User's bookmarks (optional)
 * @param {Array} history - Browser history items (optional)
 * @returns {Promise<Array>} Array of 5 personalized suggestions
 */
export async function generatePersonalizedSuggestions(config, recentSessions = [], bookmarks = [], history = []) {
  try {
    console.log('[SUGGESTIONS.JS] Starting generation with config:', {
      model: config.model,
      hasApiKey: !!config.apiKey,
      baseURL: config.baseURL
    });

    // Initialize LLM with provided config
    // initChatModel expects (modelString, config) format
    // Pass baseURL in configuration object like ai_agent.js does
    const modelString = config.model || 'openai:gpt-4o-mini';
    console.log('[SUGGESTIONS.JS] Initializing LLM with model:', modelString);
    const llm = await initChatModel(modelString, {
      apiKey: config.apiKey,
      temperature: config.temperature,
      configuration: { baseURL: config.baseURL }
    });
    console.log('[SUGGESTIONS.JS] LLM initialized successfully');

    // Prepare context from user data
    const recentQueriesContext = recentSessions
      .map((session, idx) => `${idx + 1}. "${session.query}" (${new Date(session.lastUpdated).toLocaleDateString()})`)
      .join('\n');

    const bookmarksContext = bookmarks.length > 0
      ? bookmarks.slice(0, 10).map(b => `- ${b.title}: ${b.url}`).join('\n')
      : 'No bookmarks available';

    const historyContext = history.length > 0
      ? history.slice(0, 10).map(h => `- ${h.title || h.url}`).join('\n')
      : 'No recent history available';

    // Create prompt for LLM
    const prompt = `You are a helpful AI assistant analyzing a user's browser usage patterns to suggest personalized actions.

Based on the user's recent activity, generate 5 personalized action suggestions they might want to try next.

Recent Queries:
${recentQueriesContext || 'No recent queries'}

Recent Bookmarks:
${bookmarksContext}

Recent Browsing History:
${historyContext}

Guidelines:
- Make suggestions actionable and specific
- Consider the user's interests based on their history
- Vary the types of actions (research, productivity, learning, entertainment, etc.)
- Keep titles concise (5-8 words)
- Use emojis that clearly represent each action
- Make descriptions brief but informative (10-15 words)
- Avoid duplicating recent queries - suggest new related actions instead

Generate 5 diverse, personalized suggestions that would be genuinely useful to this user.`;

    // Use withStructuredOutput for clean schema-based generation
    console.log('[SUGGESTIONS.JS] Creating LLM with structured output...');
    const structuredLLM = llm.withStructuredOutput(SuggestionSchema);

    // Invoke with structured output
    console.log('[SUGGESTIONS.JS] Invoking LLM with structured output...');
    const result = await structuredLLM.invoke(prompt);
    console.log('[SUGGESTIONS.JS] Structured output received:', result.suggestions?.length || 0);

    return result.suggestions;

  } catch (error) {
    console.error('[SUGGESTIONS.JS] Error generating personalized suggestions:', error);
    console.error('[SUGGESTIONS.JS] Error stack:', error.stack);
    throw error;
  }
}

/**
 * Get static fallback suggestions
 * @returns {Array} Array of 5 static suggestions
 */
export function getStaticSuggestions() {
  return [
    {
      emoji: "📝",
      title: "Outline a short story on a canvas",
      description: "Open a canvas and outline a 1,000-word short story"
    },
    {
      emoji: "💡",
      title: "Build a weekly meal plan and grocery list",
      description: "Create a 7-day dinner meal plan for two adults"
    },
    {
      emoji: "🚀",
      title: "Compare phone plans and pick the best deal",
      description: "Research current mobile phone plans for an individual"
    },
    {
      emoji: "🎯",
      title: "Start study mode to master a tough topic",
      description: "Learn about complex topics with interactive study mode"
    },
    {
      emoji: "🔍",
      title: "Discover trending topics",
      description: "Find out what's popular and interesting today"
    }
  ];
}
