/**
 * SkillService - Manages skills in chrome.storage.local
 * 
 * Skills follow the Agent Skills Standard (agentskills.io) compatible with
 * OpenAI Codex and Claude Code. Skills are stored as SKILL.md format with
 * YAML frontmatter for metadata and markdown body for instructions.
 * 
 * Storage key: vibe_skills
 */

// Skill interface matching Agent Skills Standard
export interface Skill {
  id: string;                    // UUID
  name: string;                  // max 64 chars, lowercase + hyphens only
  description: string;           // max 1024 chars
  content: string;               // Full SKILL.md content (frontmatter + body)
  enabled: boolean;
  isBuiltIn: boolean;
  createdAt: number;
  updatedAt: number;
}

// Parsed skill with extracted body (for agent use)
export interface ParsedSkill extends Skill {
  body: string;                  // Markdown body without frontmatter
}

// Validation constraints from Agent Skills Standard
const MAX_NAME_LENGTH = 64;
const MAX_DESCRIPTION_LENGTH = 1024;
const NAME_PATTERN = /^[a-z0-9]+(-[a-z0-9]+)*$/;

// Storage key
const STORAGE_KEY = 'vibe_skills';

/**
 * Parse SKILL.md content to extract YAML frontmatter and markdown body
 */
export function parseSkillContent(content: string): { 
  name?: string; 
  description?: string; 
  body: string;
  metadata?: Record<string, unknown>;
} {
  const frontmatterMatch = content.match(/^---\s*\n([\s\S]*?)\n---\s*\n?([\s\S]*)$/);
  
  if (!frontmatterMatch) {
    // No frontmatter, treat entire content as body
    return { body: content.trim() };
  }

  const [, frontmatter, body] = frontmatterMatch;
  const result: { name?: string; description?: string; body: string; metadata?: Record<string, unknown> } = {
    body: body.trim()
  };

  // Simple YAML parsing for key fields (avoiding external dependencies)
  const lines = frontmatter.split('\n');
  let currentKey = '';
  let currentValue = '';
  let inMultiline = false;

  for (const line of lines) {
    // Check for key: value pattern
    const keyMatch = line.match(/^(\w+(?:-\w+)*):\s*(.*)$/);
    
    if (keyMatch && !inMultiline) {
      // Save previous key if exists
      if (currentKey) {
        assignValue(result, currentKey, currentValue.trim());
      }
      
      currentKey = keyMatch[1];
      currentValue = keyMatch[2];
      
      // Check for multiline indicator
      if (currentValue === '|' || currentValue === '>') {
        inMultiline = true;
        currentValue = '';
      }
    } else if (inMultiline && line.startsWith('  ')) {
      // Continuation of multiline value
      currentValue += (currentValue ? '\n' : '') + line.slice(2);
    } else if (inMultiline && !line.startsWith('  ')) {
      // End of multiline
      inMultiline = false;
      if (currentKey) {
        assignValue(result, currentKey, currentValue.trim());
      }
      // Process this line as new key
      const newKeyMatch = line.match(/^(\w+(?:-\w+)*):\s*(.*)$/);
      if (newKeyMatch) {
        currentKey = newKeyMatch[1];
        currentValue = newKeyMatch[2];
      }
    }
  }
  
  // Don't forget last key
  if (currentKey) {
    assignValue(result, currentKey, currentValue.trim());
  }

  return result;
}

function assignValue(result: Record<string, unknown>, key: string, value: string): void {
  if (key === 'name') {
    result.name = value;
  } else if (key === 'description') {
    result.description = value;
  } else {
    // Store other fields in metadata
    if (!result.metadata) {
      result.metadata = {};
    }
    (result.metadata as Record<string, unknown>)[key] = value;
  }
}

/**
 * Generate SKILL.md content from name, description, and body
 */
export function generateSkillContent(name: string, description: string, body: string): string {
  return `---
name: ${name}
description: ${description}
---

${body}`;
}

/**
 * Validate skill name according to Agent Skills Standard
 */
export function validateSkillName(name: string): { valid: boolean; error?: string } {
  if (!name) {
    return { valid: false, error: 'Name is required' };
  }
  if (name.length > MAX_NAME_LENGTH) {
    return { valid: false, error: `Name must be ${MAX_NAME_LENGTH} characters or less` };
  }
  if (!NAME_PATTERN.test(name)) {
    return { valid: false, error: 'Name must be lowercase letters, numbers, and hyphens only. Cannot start/end with hyphen or have consecutive hyphens.' };
  }
  return { valid: true };
}

/**
 * Validate skill description
 */
export function validateSkillDescription(description: string): { valid: boolean; error?: string } {
  if (!description) {
    return { valid: false, error: 'Description is required' };
  }
  if (description.length > MAX_DESCRIPTION_LENGTH) {
    return { valid: false, error: `Description must be ${MAX_DESCRIPTION_LENGTH} characters or less` };
  }
  return { valid: true };
}

/**
 * Validation issue with severity level
 */
export interface ValidationIssue {
  type: 'error' | 'warning';
  field: 'frontmatter' | 'name' | 'description' | 'body' | 'format';
  message: string;
  line?: number;
}

/**
 * Complete validation result for skill content
 */
export interface SkillValidationResult {
  valid: boolean;
  issues: ValidationIssue[];
  parsed: {
    name?: string;
    description?: string;
    body: string;
    hasFrontmatter: boolean;
  };
}

/**
 * Validate complete skill content with detailed feedback
 * Returns all validation issues for real-time editor feedback
 */
export function validateSkillContent(content: string): SkillValidationResult {
  const issues: ValidationIssue[] = [];
  
  // Check for empty content
  if (!content || !content.trim()) {
    return {
      valid: false,
      issues: [{ type: 'error', field: 'format', message: 'Skill content is empty' }],
      parsed: { body: '', hasFrontmatter: false }
    };
  }

  // Check for frontmatter
  const frontmatterMatch = content.match(/^---\s*\n([\s\S]*?)\n---\s*\n?([\s\S]*)$/);
  const hasFrontmatter = !!frontmatterMatch;
  
  if (!hasFrontmatter) {
    issues.push({
      type: 'error',
      field: 'frontmatter',
      message: 'Missing YAML frontmatter. Skill must start with --- on the first line.',
      line: 1
    });
    return {
      valid: false,
      issues,
      parsed: { body: content.trim(), hasFrontmatter: false }
    };
  }

  // Parse the content
  const parsed = parseSkillContent(content);
  
  // Validate name
  if (!parsed.name) {
    issues.push({
      type: 'error',
      field: 'name',
      message: 'Missing required field: name',
      line: 2
    });
  } else {
    const nameValidation = validateSkillName(parsed.name);
    if (!nameValidation.valid) {
      issues.push({
        type: 'error',
        field: 'name',
        message: nameValidation.error!,
        line: 2
      });
    }
  }

  // Validate description
  if (!parsed.description) {
    issues.push({
      type: 'error',
      field: 'description',
      message: 'Missing required field: description',
      line: 3
    });
  } else {
    const descValidation = validateSkillDescription(parsed.description);
    if (!descValidation.valid) {
      issues.push({
        type: 'error',
        field: 'description',
        message: descValidation.error!,
        line: 3
      });
    }
  }

  // Validate body
  if (!parsed.body || parsed.body.trim().length === 0) {
    issues.push({
      type: 'warning',
      field: 'body',
      message: 'Skill body is empty. Add instructions for the agent.'
    });
  } else if (parsed.body.trim().length < 20) {
    issues.push({
      type: 'warning',
      field: 'body',
      message: 'Skill body is very short. Consider adding more detailed instructions.'
    });
  }

  // Check frontmatter closing
  const lines = content.split('\n');
  if (lines[0].trim() !== '---') {
    issues.push({
      type: 'error',
      field: 'frontmatter',
      message: 'Frontmatter must start with --- on the first line',
      line: 1
    });
  }

  // Find closing ---
  let closingLineIndex = -1;
  for (let i = 1; i < lines.length; i++) {
    if (lines[i].trim() === '---') {
      closingLineIndex = i;
      break;
    }
  }
  
  if (closingLineIndex === -1) {
    issues.push({
      type: 'error',
      field: 'frontmatter',
      message: 'Missing closing --- for frontmatter'
    });
  }

  return {
    valid: issues.filter(i => i.type === 'error').length === 0,
    issues,
    parsed: {
      name: parsed.name,
      description: parsed.description,
      body: parsed.body,
      hasFrontmatter
    }
  };
}

/**
 * SkillService class for CRUD operations
 */
export class SkillService {
  private static instance: SkillService;

  private constructor() {}

  static getInstance(): SkillService {
    if (!SkillService.instance) {
      SkillService.instance = new SkillService();
    }
    return SkillService.instance;
  }

  /**
   * Get chrome storage (works in extension context)
   */
  private getStorage(): typeof chrome.storage.local | null {
    if (typeof chrome !== 'undefined' && chrome.storage?.local) {
      return chrome.storage.local;
    }
    return null;
  }

  /**
   * Get all skills from storage
   */
  async getAll(): Promise<Skill[]> {
    const storage = this.getStorage();
    if (!storage) {
      console.warn('SkillService: chrome.storage not available');
      return [];
    }

    return new Promise((resolve) => {
      storage.get([STORAGE_KEY]).then((result: { [key: string]: unknown }) => {
        const skills = (result[STORAGE_KEY] as Skill[]) || [];
        resolve(skills);
      });
    });
  }

  /**
   * Get enabled skills only
   */
  async getEnabled(): Promise<Skill[]> {
    const all = await this.getAll();
    return all.filter(s => s.enabled);
  }

  /**
   * Get a skill by ID
   */
  async getById(id: string): Promise<Skill | null> {
    const all = await this.getAll();
    return all.find(s => s.id === id) || null;
  }

  /**
   * Get a skill by name
   */
  async getByName(name: string): Promise<Skill | null> {
    const all = await this.getAll();
    return all.find(s => s.name === name) || null;
  }

  /**
   * Get a parsed skill (with body extracted) by name
   */
  async getParsedByName(name: string): Promise<ParsedSkill | null> {
    const skill = await this.getByName(name);
    if (!skill) return null;
    
    const parsed = parseSkillContent(skill.content);
    return {
      ...skill,
      body: parsed.body
    };
  }

  /**
   * Create a new skill
   */
  async create(content: string, options?: { enabled?: boolean; isBuiltIn?: boolean }): Promise<Skill> {
    const parsed = parseSkillContent(content);
    
    if (!parsed.name) {
      throw new Error('Skill must have a name in frontmatter');
    }
    if (!parsed.description) {
      throw new Error('Skill must have a description in frontmatter');
    }

    const nameValidation = validateSkillName(parsed.name);
    if (!nameValidation.valid) {
      throw new Error(nameValidation.error);
    }

    const descValidation = validateSkillDescription(parsed.description);
    if (!descValidation.valid) {
      throw new Error(descValidation.error);
    }

    // Check for duplicate name
    const existing = await this.getByName(parsed.name);
    if (existing) {
      throw new Error(`Skill with name "${parsed.name}" already exists`);
    }

    const now = Date.now();
    const skill: Skill = {
      id: crypto.randomUUID(),
      name: parsed.name,
      description: parsed.description,
      content,
      enabled: options?.enabled ?? true,
      isBuiltIn: options?.isBuiltIn ?? false,
      createdAt: now,
      updatedAt: now
    };

    const all = await this.getAll();
    all.push(skill);
    await this.saveAll(all);

    return skill;
  }

  /**
   * Update an existing skill
   */
  async update(id: string, content: string): Promise<Skill> {
    const all = await this.getAll();
    const index = all.findIndex(s => s.id === id);
    
    if (index === -1) {
      throw new Error(`Skill with ID "${id}" not found`);
    }

    const parsed = parseSkillContent(content);
    
    if (!parsed.name) {
      throw new Error('Skill must have a name in frontmatter');
    }
    if (!parsed.description) {
      throw new Error('Skill must have a description in frontmatter');
    }

    const nameValidation = validateSkillName(parsed.name);
    if (!nameValidation.valid) {
      throw new Error(nameValidation.error);
    }

    const descValidation = validateSkillDescription(parsed.description);
    if (!descValidation.valid) {
      throw new Error(descValidation.error);
    }

    // Check for duplicate name (excluding current skill)
    const existing = await this.getByName(parsed.name);
    if (existing && existing.id !== id) {
      throw new Error(`Skill with name "${parsed.name}" already exists`);
    }

    const skill = all[index];
    skill.name = parsed.name;
    skill.description = parsed.description;
    skill.content = content;
    skill.updatedAt = Date.now();

    await this.saveAll(all);
    return skill;
  }

  /**
   * Toggle skill enabled status
   */
  async setEnabled(id: string, enabled: boolean): Promise<Skill> {
    const all = await this.getAll();
    const skill = all.find(s => s.id === id);
    
    if (!skill) {
      throw new Error(`Skill with ID "${id}" not found`);
    }

    skill.enabled = enabled;
    skill.updatedAt = Date.now();
    await this.saveAll(all);
    
    return skill;
  }

  /**
   * Delete a skill
   */
  async delete(id: string): Promise<void> {
    const all = await this.getAll();
    const index = all.findIndex(s => s.id === id);
    
    if (index === -1) {
      throw new Error(`Skill with ID "${id}" not found`);
    }

    if (all[index].isBuiltIn) {
      throw new Error('Cannot delete built-in skills');
    }

    all.splice(index, 1);
    await this.saveAll(all);
  }

  /**
   * Import skills from JSON (for import/export functionality)
   */
  async import(skills: Skill[], mode: 'replace' | 'merge' = 'merge'): Promise<number> {
    let all = mode === 'replace' ? [] : await this.getAll();
    let imported = 0;

    for (const skill of skills) {
      // Validate
      const nameValidation = validateSkillName(skill.name);
      const descValidation = validateSkillDescription(skill.description);
      
      if (!nameValidation.valid || !descValidation.valid) {
        console.warn(`Skipping invalid skill: ${skill.name}`);
        continue;
      }

      if (mode === 'merge') {
        // Check for existing skill with same name
        const existingIndex = all.findIndex(s => s.name === skill.name);
        if (existingIndex !== -1) {
          // Replace existing
          all[existingIndex] = {
            ...skill,
            id: all[existingIndex].id, // Keep original ID
            updatedAt: Date.now()
          };
        } else {
          // Add new
          all.push({
            ...skill,
            id: crypto.randomUUID(),
            createdAt: Date.now(),
            updatedAt: Date.now()
          });
        }
      } else {
        all.push({
          ...skill,
          id: crypto.randomUUID(),
          createdAt: Date.now(),
          updatedAt: Date.now()
        });
      }
      imported++;
    }

    await this.saveAll(all);
    return imported;
  }

  /**
   * Export all skills (for import/export functionality)
   */
  async export(): Promise<Skill[]> {
    return this.getAll();
  }

  /**
   * Initialize with built-in skills if none exist
   */
  async initializeBuiltIns(builtInSkills: string[]): Promise<void> {
    const all = await this.getAll();
    
    // Skip if we already have skills
    if (all.length > 0) {
      return;
    }

    for (const content of builtInSkills) {
      try {
        await this.create(content, { enabled: true, isBuiltIn: true });
      } catch (error) {
        console.warn('Failed to create built-in skill:', error);
      }
    }
  }

  /**
   * Save all skills to storage
   */
  private async saveAll(skills: Skill[]): Promise<void> {
    const storage = this.getStorage();
    if (!storage) {
      console.warn('SkillService: chrome.storage not available');
      return;
    }

    return new Promise<void>((resolve) => {
      storage.set({ [STORAGE_KEY]: skills }).then(() => {
        resolve();
      });
    });
  }

  /**
   * Get skill summaries for system prompt injection
   * Returns only name and description for each enabled skill
   */
  async getSkillSummaries(): Promise<Array<{ name: string; description: string }>> {
    const enabled = await this.getEnabled();
    return enabled.map(s => ({
      name: s.name,
      description: s.description
    }));
  }
}

// Export singleton instance
export const skillService = SkillService.getInstance();
