/**
 * Shared message utilities for serialization/deserialization
 * Used by both ReactGraph StateSerializer and Chat4 messageUtils
 */

import { HumanMessage, AIMessage, SystemMessage, ToolMessage } from "@langchain/core/messages";

// Map of constructor names to constructor functions
const MESSAGE_CONSTRUCTORS = {
  HumanMessage,
  AIMessage,
  SystemMessage,
  ToolMessage
};

/**
 * Deserialize a single LangChain message from serialized format
 * Handles both simple format {type: "HumanMessage", content: "..."}
 * and LangChain native format {lc: 1, type: "constructor", id: [..., "HumanMessage"], kwargs: {...}}
 * @param {Object} msg - Serialized message object
 * @returns {BaseMessage|null} LangChain message object or null if invalid
 */
export function deserializeMessage(msg) {
  if (!msg) {
    return null;
  }

  // Handle LangChain native serialization format: {lc: 1, type: "constructor", id: [...], kwargs: {...}}
  if (msg.lc === 1 && msg.type === 'constructor' && Array.isArray(msg.id) && msg.kwargs) {
    // Extract message type from id array (e.g., ["langchain_core", "messages", "HumanMessage"])
    const messageType = msg.id[msg.id.length - 1];
    const Constructor = MESSAGE_CONSTRUCTORS[messageType];
    if (!Constructor) {
      // Skip non-message types silently (they're expected in allMessages)
      return null;
    }

    try {
      const { content, additional_kwargs, response_metadata, tool_calls, tool_call_id, name } = msg.kwargs;

      if (messageType === 'ToolMessage') {
        const message = new Constructor(content, tool_call_id, name);
        if (additional_kwargs) {
          message.additional_kwargs = additional_kwargs;
        }
        return message;
      }

      const result = new Constructor(content, additional_kwargs || {});
      if (response_metadata) {
        result.response_metadata = response_metadata;
      }
      if (tool_calls) {
        result.tool_calls = tool_calls;
      }
      return result;
    } catch (error) {
      console.error('[messageUtils] Failed to deserialize LangChain format:', error);
      return null;
    }
  }

  // Handle simple format: {type: "HumanMessage", content: "...", ...}
  if (!msg.type) {
    return null;
  }

  const Constructor = MESSAGE_CONSTRUCTORS[msg.type];
  if (!Constructor) {
    // Log only for unexpected types (not "constructor" which is handled above)
    if (msg.type !== 'constructor') {
      console.log('[messageUtils] No constructor for type:', msg.type);
    }
    return null;
  }

  // LangChain 0.3.x uses positional parameters: Constructor(content, additional_kwargs)
  // Special handling for ToolMessage which needs (content, tool_call_id, name)
  try {
    if (msg.type === 'ToolMessage') {
      const message = new Constructor(msg.content, msg.tool_call_id, msg.name);
      if (msg.additional_kwargs) {
        message.additional_kwargs = msg.additional_kwargs;
      }
      return message;
    }

    // For all other message types: HumanMessage, AIMessage, SystemMessage
    const result = new Constructor(msg.content, msg.additional_kwargs || {});
    return result;
  } catch (error) {
    console.error('[messageUtils] Failed to deserialize message:', error, msg);
    return null;
  }
}

/**
 * Deserialize an array of LangChain messages from serialized format
 * @param {Array} serialized - Array of serialized message objects
 * @returns {Array} Array of LangChain message objects
 */
export function deserializeMessages(serialized) {
  if (!serialized || !Array.isArray(serialized)) return [];

  return serialized.map(msg => deserializeMessage(msg)).filter(msg => msg !== null);
}