/**
 * Markdown Page Extractor
 * Extracts page content as markdown with indexed interactive elements
 * Conservative visibility filtering - only skips display:none and visibility:hidden
 */

class MarkdownPageExtractor {
  /**
   * Extract indexed page content with optional visual highlighting
   * @param {Object} options Configuration options
   * @returns {Object} Extracted content with indexed elements
   */
  extractContent(options = {}) {
    const {
      showHighlights = true,
      highlightDuration = 30000,
      maxElements = 1000,
      includeInteractive = true,
      includeChanges = true
    } = options;

    // Extract all interactive elements
    const { elements, domElements } = this.extractInteractiveElements({
      maxElements
    });

    // Index elements
    const indexedElements = this.assignIndices(elements, domElements, maxElements);

    // Build content with indexed elements inline
    const content = this.buildIndexedContent(document.body, indexedElements);

    // Cache DOM elements for later use
    let cacheId = null;
    if (typeof window !== 'undefined' && window.ElementCache) {
      cacheId = window.ElementCache.store(indexedElements.domElements);
      // Store cache ID for highlighting
      this.lastCacheId = cacheId;
    }

    // Apply visual highlighting if requested
    if (showHighlights) {
        this.highlightElements(indexedElements.elements, highlightDuration);
    }

    return {
      content,
      elementCount: indexedElements.elements.length,
      elements: indexedElements.elements,
      domElements: indexedElements.domElements, // Add DOM elements for backward compatibility
      cacheId, // Include cache ID for later retrieval
      metadata: {
        title: document.title,
        url: window.location.href,
        timestamp: new Date().toISOString()
      }
    };
  }

  /**
   * Extract all interactive elements from the page
   */
  extractInteractiveElements(options) {
    const { maxElements } = options;
    const elements = [];
    const domElements = [];

    // Define selectors for interactive elements
    const interactiveSelectors = [
      'a[href]',
      'button',
      'input',
      'select',
      'textarea',
      '[role="button"]',
      '[role="link"]',
      '[role="checkbox"]',
      '[role="radio"]',
      '[role="combobox"]',
      '[role="listbox"]',
      '[role="option"]',  // Important for dropdowns
      '[role="menuitem"]',
      '[role="tab"]',
      '[role="switch"]',  // Toggle controls
      '[role="slider"]',  // Range controls
      '[role="menuitemcheckbox"]',
      '[role="menuitemradio"]',
      '[role="treeitem"]',
      '[onclick]',
      '[tabindex]:not([tabindex="-1"]):not(c-wiz)',  // Exclude c-wiz containers
      '[contenteditable="true"]',
      // Additional selectors for better dropdown detection
      '.autocomplete-suggestion',
      '.dropdown-item',
      '[data-suggestion]',
      'li[role="option"]',
      'div[role="option"]',
      // Modern SPA patterns
      '[data-action]',
      '[data-clickable]',
      '[data-link]',
      '[aria-haspopup]',
      // Framework-specific handlers
      '[ng-click]',
      '[v-on\\:click]',
      // CSS-based clickability
      '.clickable',
      '.interactive'
    ];

    // Query all interactive elements
    const allElements = document.querySelectorAll(interactiveSelectors.join(', '));

    for (const element of allElements) {
      if (elements.length >= maxElements) break;

      const elementInfo = this.extractElementInfo(element);
      if (elementInfo) {
        elements.push(elementInfo);
        domElements.push(element);
      }
    }

    return { elements, domElements };
  }

  /**
   * Extract information from a single element
   */
  extractElementInfo(element) {
    const tagName = element.tagName.toLowerCase();

    // Skip container elements that shouldn't be interactive
    if (tagName === 'c-wiz' || tagName === 'g-scrolling-carousel') {
      return null;
    }

    const rect = element.getBoundingClientRect();

    // Get text content - UNIVERSAL approach that works on all pages
    let text = '';

    if (tagName === 'input' || tagName === 'textarea') {
      // Form inputs: use value
      text = element.value || '';
    } else if (tagName === 'select') {
      // Select dropdowns: use selected option text
      const selectedOption = element.options[element.selectedIndex];
      text = selectedOption ? selectedOption.text : '';
    } else {
      // ALL other elements: use innerText which shows visible rendered text
      // innerText respects CSS visibility and gives text as user sees it
      // Falls back to textContent if innerText is not available
      text = element.innerText?.trim() || element.textContent?.trim() || '';
    }

    // Limit text length
    if (text.length > 100) {
      text = text.substring(0, 97) + '...';
    }

    // Check if element is in a form
    const isInForm = !!element.closest('form');

    const elementInfo = {
      tagName,
      text,
      attributes: this.extractAttributes(element),
      rect: {
        top: rect.top,
        left: rect.left,
        width: rect.width,
        height: rect.height
      },
      isInForm
    };

    // Calculate and add score (pass DOM element for advanced scoring)
    elementInfo.score = this.calculateElementScore(elementInfo, element);

    return elementInfo;
  }

  /**
   * Extract relevant attributes from element
   */
  extractAttributes(element) {
    const attrs = {};
    // Note: href is intentionally excluded to minimize context usage
    // Since every element is indexed, the agent clicks by index rather than URL
    const relevantAttrs = [
      'type', 'name', 'id', 'class', 'placeholder', 'value',
      'src', 'alt', 'title', 'role', 'aria-label',
      'aria-describedby', 'aria-expanded', 'aria-selected',
      'disabled', 'readonly', 'required', 'checked',
      'contenteditable', 'data-testid'
    ];

    for (const attr of relevantAttrs) {
      const value = element.getAttribute(attr);
      if (value !== null && value !== '') {
        attrs[attr] = value;
      }
    }

    return attrs;
  }

  /**
   * Assign indices to elements
   */
  assignIndices(elements, domElements, maxElements) {
    const indexedElements = [];
    const elementByNode = new Map();
    const indexedDomElements = [];

    const limit = Math.min(elements.length, maxElements);
    for (let i = 0; i < limit; i++) {
      const element = elements[i];
      const domElement = domElements[i];

      element.index = i;
      indexedElements.push(element);
      indexedDomElements.push(domElement);
      elementByNode.set(domElement, element);
    }

    return { elements: indexedElements, domElements: indexedDomElements, elementByNode };
  }

  /**
   * Build content with indexed elements inline
   */
  buildIndexedContent(rootNode, indexedData) {
    const { elementByNode } = indexedData;
    const parts = [];

    // Add header explaining the scoring system
    parts.push('// Interactive elements with scores [index:score]\n');
    parts.push('// Higher scores = better click targets (visible:10, enabled:10, in-form:20, button:5, submit:15)\n');
    parts.push('// When multiple similar elements exist, prefer those with higher scores\n\n');


    const traverse = (node) => {
      // Skip script, style, noscript, and code elements
      // Code tags often contain hidden JSON state (e.g., LinkedIn's `<code style="display:none">`)
      if (node.tagName && ['SCRIPT', 'STYLE', 'NOSCRIPT', 'CODE'].includes(node.tagName)) {
        return;
      }

      // Check if this node is an indexed element
      const indexedElement = elementByNode.get(node);
      if (indexedElement) {
        // Add indexed element in HTML format
        parts.push(this.formatIndexedElement(indexedElement));
        parts.push(' ');
        // IMPORTANT: Continue processing children to extract ALL text content
        // This matches browser-use behavior - we want all text, not just interactive elements
      }

      // Special handling for tables to preserve structure
      if (node.tagName && node.tagName.toLowerCase() === 'table') {
        parts.push('\n');
        this.extractTableContent(node, parts, elementByNode);
        parts.push('\n');
        return; // Don't traverse table children normally
      }

      // Process children
      for (const child of node.childNodes) {
        if (child.nodeType === Node.TEXT_NODE) {
          const text = child.textContent.trim();
          if (text) {
            // Mark any user-selected text with **>><<** markers
            const markedText = this.markSelectedText(child, text);
            parts.push(markedText);
            parts.push(' ');
          }
        } else if (child.nodeType === Node.ELEMENT_NODE) {
          // Only skip explicitly hidden elements - be conservative
          // Don't skip opacity:0 or size-based checks as they may be wrong
          const style = window.getComputedStyle(child);
          if (style.display === 'none' || style.visibility === 'hidden') {
            continue;
          }

          // Handle block elements with smarter newline management
          const tagName = child.tagName.toLowerCase();
          const isBlockElement = ['div', 'p', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'li', 'section', 'article', 'header', 'footer', 'main', 'nav', 'tr', 'td', 'th'].includes(tagName);
          const lastChar = parts.length > 0 ? parts[parts.length - 1].slice(-1) : '';
          const needsNewlineBefore = isBlockElement && lastChar !== '\n';

          // Add line break before block elements only if needed
          if (needsNewlineBefore) {
            parts.push('\n');
          }

          // Special handling for br - just ensure one newline
          if (tagName === 'br') {
            if (lastChar !== '\n') {
              parts.push('\n');
            }
            continue; // Skip processing children
          }

          // Add markdown headers
          if (tagName.match(/^h[1-6]$/)) {
            const level = parseInt(tagName[1]);
            parts.push('#'.repeat(level) + ' ');
          }

          // Recursively process
          traverse(child);

          // Add line break after block elements only if content was added
          const currentLastChar = parts.length > 0 ? parts[parts.length - 1].slice(-1) : '';
          if (isBlockElement && currentLastChar !== '\n') {
            parts.push('\n');
          }
        }
      }
    };

    traverse(rootNode);

    // Clean up the final content
    let content = parts.join('')
      .replace(/\n{3,}/g, '\n\n')  // Replace 3+ newlines with 2
      .replace(/\s*\n\s*\n\s*/g, '\n\n')  // Clean up whitespace around double newlines
      .replace(/^\n+/, '')  // Remove leading newlines
      .replace(/\n+$/, '')  // Remove trailing newlines
      .trim();

    return content;
  }

  /**
   * Extract table content in a structured way
   */
  extractTableContent(tableNode, parts, elementByNode) {
    // Check if this table is actually used for layout (not data)
    // Layout tables typically have no headers and contain diverse content
    const isLayoutTable = this.isLayoutTable(tableNode);

    if (isLayoutTable) {
      // For layout tables, just extract all visible text without table formatting
      this.extractLayoutTableContent(tableNode, parts, elementByNode);
      return;
    }

    // For data tables, extract as markdown table
    const rows = tableNode.querySelectorAll('tr');

    // First pass: check if this is a header row
    const headers = [];
    const firstRow = rows[0];
    if (firstRow) {
      const headerCells = firstRow.querySelectorAll('th');
      if (headerCells.length > 0) {
        headerCells.forEach(th => {
          headers.push(this.extractCellText(th, elementByNode).trim());
        });
        if (headers.length > 0) {
          parts.push('| ' + headers.join(' | ') + ' |\n');
          parts.push('|' + headers.map(() => '---').join('|') + '|\n');
        }
      }
    }

    // Process all rows
    rows.forEach((row, rowIndex) => {
      const cells = row.querySelectorAll('td, th');
      if (cells.length === 0) return;

      const rowData = [];
      cells.forEach(cell => {
        // Get all text content from cell, including indexed elements
        const cellText = this.extractCellText(cell, elementByNode).trim();
        rowData.push(cellText);
      });

      if (rowData.some(data => data.trim())) {
        // If this is the first row and we didn't find headers, treat it as data
        if (rowIndex === 0 && headers.length === 0) {
          // Check if first row might be headers (no th elements but looks like headers)
          const possibleHeaders = rowData.every(cell =>
            cell.length < 20 && !cell.match(/\d+\.?\d*%/) && !cell.match(/\$[\d,]+/)
          );

          if (possibleHeaders) {
            parts.push('| ' + rowData.join(' | ') + ' |\n');
            parts.push('|' + rowData.map(() => '---').join('|') + '|\n');
            return;
          }
        }

        parts.push('| ' + rowData.join(' | ') + ' |\n');
      }
    });
  }

  /**
   * Check if a table is used for layout rather than data
   */
  isLayoutTable(tableNode) {
    // Layout tables typically:
    // 1. Have no <th> elements
    // 2. Have inconsistent column counts
    // 3. Contain non-tabular content (links, navigation, complex nested structures)
    // 4. Don't have borders or data-table attributes

    const hasHeaders = tableNode.querySelector('th') !== null;
    const rows = tableNode.querySelectorAll('tr');

    // Check if it has table headers
    if (hasHeaders) {
      return false; // Likely a data table
    }

    // Check for consistent column count (data tables usually have consistent columns)
    const columnCounts = new Set();
    rows.forEach(row => {
      const cellCount = row.querySelectorAll('td').length;
      if (cellCount > 0) {
        columnCounts.add(cellCount);
      }
    });

    // If column counts vary significantly, it's likely a layout table
    const hasInconsistentColumns = columnCounts.size > 3;

    // Check for typical layout patterns
    const hasNestedTables = tableNode.querySelector('table') !== null;
    const hasComplexContent = tableNode.querySelectorAll('a').length > 20; // Many links suggest navigation/layout

    // Check for data table attributes
    const hasDataAttributes = tableNode.hasAttribute('data-table') ||
                             tableNode.classList.contains('data-table') ||
                             tableNode.classList.contains('table-striped');

    return !hasDataAttributes && (hasInconsistentColumns || hasNestedTables || hasComplexContent);
  }

  /**
   * Extract content from layout tables (like HN) without table formatting
   */
  extractLayoutTableContent(tableNode, parts, elementByNode) {
    // For layout tables, traverse and extract all content linearly
    const rows = tableNode.querySelectorAll('tr');

    rows.forEach(row => {
      // Extract all visible text from the row
      const rowText = this.extractVisibleText(row, elementByNode).trim();

      if (rowText) {
        // Add the content with appropriate formatting
        // Detect if this looks like a list item (starts with number)
        if (/^\d+\./.test(rowText)) {
          parts.push('\n' + rowText + '\n');
        } else if (rowText.length > 0) {
          parts.push(rowText + '\n');
        }
      }
    });
  }

  /**
   * Extract all visible text from a node
   */
  extractVisibleText(node, elementByNode) {
    let text = '';

    const extractText = (n) => {
      // Check for indexed elements
      if (elementByNode && elementByNode.get(n)) {
        const indexedEl = elementByNode.get(n);
        text += this.formatIndexedElement(indexedEl) + ' ';
      }

      if (n.nodeType === Node.TEXT_NODE) {
        const t = n.textContent.trim();
        if (t) {
          text += t + ' ';
        }
      } else if (n.nodeType === Node.ELEMENT_NODE) {
        // Skip hidden elements
        const style = window.getComputedStyle(n);
        if (style.display === 'none' || style.visibility === 'hidden') {
          return;
        }

        // Process all child nodes
        for (const child of n.childNodes) {
          extractText(child);
        }
      }
    };

    extractText(node);
    return text.trim();
  }

  /**
   * Extract all text from a table cell, including nested elements
   */
  extractCellText(cell, elementByNode) {
    let text = '';

    const extractText = (node) => {
      // Check for indexed elements
      if (elementByNode && elementByNode.get(node)) {
        const indexedEl = elementByNode.get(node);
        text += this.formatIndexedElement(indexedEl) + ' ';
        return; // Don't process children of indexed elements
      }

      if (node.nodeType === Node.TEXT_NODE) {
        text += node.textContent;
      } else if (node.nodeType === Node.ELEMENT_NODE) {
        // Skip hidden elements
        const style = window.getComputedStyle(node);
        if (style.display === 'none' || style.visibility === 'hidden') {
          return;
        }

        // Process all child nodes
        for (const child of node.childNodes) {
          extractText(child);
        }
      }
    };

    extractText(cell);
    return text;
  }

  /**
   * Calculate element score based on visibility and context
   */
  calculateElementScore(element, domElement = null) {
    let score = 0;

    // Check visibility
    const rect = element.rect || {};
    const isVisible = rect.width > 0 && rect.height > 0;
    if (isVisible) score += 10;

    // Check if enabled (not disabled)
    const isDisabled = element.attributes?.disabled === 'true' ||
                      element.attributes?.['aria-disabled'] === 'true';
    if (!isDisabled) score += 10;

    // Check if in form
    if (element.isInForm || element.attributes?.form || element.tagName === 'fieldset') {
      score += 20;
    }

    // Element type scoring
    if (element.tagName === 'button') score += 5;
    if (element.attributes?.type === 'submit') score += 15;

    // Has descriptive text
    if (element.text || element.attributes?.['aria-label']) score += 3;

    // Context-based scoring
    if (element.attributes?.['aria-haspopup']) score += 5;  // Menu triggers
    if (element.attributes?.role === 'dialog') score += 8;  // Modal triggers
    if (element.attributes?.role === 'switch') score += 4;  // Toggle controls

    // Position-based scoring (above-the-fold elements more important)
    if (typeof window !== 'undefined' && rect.top < window.innerHeight) {
      score += 7;
    }

    // Interactive pattern indicators
    if (element.attributes?.['data-action']) score += 4;
    if (element.attributes?.['data-clickable']) score += 4;

    // Check for pointer cursor (indicates developer intent for interactivity)
    if (domElement) {
      try {
        const style = window.getComputedStyle(domElement);
        if (style.cursor === 'pointer') score += 6;
      } catch (e) {
        // Ignore if element no longer in DOM
      }
    }

    // Penalize deeply nested elements (often duplicates or decorative)
    if (domElement) {
      const depth = this.getElementDepth(domElement);
      if (depth > 10) score -= 3;
      if (depth > 15) score -= 3; // Additional penalty for very deep nesting
    }

    return Math.max(0, score); // Ensure non-negative
  }

  /**
   * Get DOM tree depth of an element
   */
  getElementDepth(element) {
    let depth = 0;
    let parent = element.parentElement;
    while (parent && depth < 20) {
      depth++;
      parent = parent.parentElement;
    }
    return depth;
  }

  /**
   * Format indexed element as HTML
   */
  formatIndexedElement(element) {
    // Calculate score if not already present
    const score = element.score !== undefined ? element.score : this.calculateElementScore(element);

    let html = `[${element.index}:${score}] `;
    html += `<${element.tagName}`;

    // Add key attributes
    // Note: href excluded - agent clicks by index, not URL
    const importantAttrs = ['type', 'placeholder', 'aria-label', 'name', 'id', 'role', 'value', 'contenteditable'];
    for (const attr of importantAttrs) {
      if (element.attributes[attr]) {
        html += ` ${attr}="${element.attributes[attr]}"`;
      }
    }

    // Self-closing tags
    if (['input', 'img', 'br', 'hr', 'meta', 'link'].includes(element.tagName)) {
      html += ' />';
    } else if (element.text) {
      html += `>${element.text}</${element.tagName}>`;
    } else {
      html += `></${element.tagName}>`;
    }

    return html;
  }

  /**
   * Apply visual highlighting to indexed elements using absolute positioning
   */
  highlightElements(elements, duration) {
    // Clean up any existing highlights
    this.cleanupHighlights();

    // Color palette for different indices (cycling colors)
    const colorPalette = [
      { border: '#FF4444', bg: 'rgba(255, 68, 68, 0.15)', label: '#FF4444' },   // Red
      { border: '#44FF44', bg: 'rgba(68, 255, 68, 0.15)', label: '#44FF44' },   // Green
      { border: '#4444FF', bg: 'rgba(68, 68, 255, 0.15)', label: '#4444FF' },   // Blue
      { border: '#FFAA44', bg: 'rgba(255, 170, 68, 0.15)', label: '#FFAA44' },  // Orange
      { border: '#FF44FF', bg: 'rgba(255, 68, 255, 0.15)', label: '#FF44FF' },  // Magenta
      { border: '#44FFFF', bg: 'rgba(68, 255, 255, 0.15)', label: '#44FFFF' },  // Cyan
      { border: '#FFFF44', bg: 'rgba(255, 255, 68, 0.15)', label: '#DDDD00' },  // Yellow (darker label)
      { border: '#FF8844', bg: 'rgba(255, 136, 68, 0.15)', label: '#FF8844' },  // Light Orange
    ];

    elements.forEach((element, index) => {
      try {
        // Try to find the actual DOM element using cached reference or fallback methods
        const domElement = this.findDomElementForIndex(element.index);

        if (!domElement) {
          console.warn(`Could not find DOM element for index ${element.index}`);
          return;
        }

        // Get absolute position relative to document
        const rect = this.getAbsolutePosition(domElement);

        // Skip if element is not visible
        if (rect.width === 0 || rect.height === 0) return;

        // Select color based on index (cycle through palette)
        const colors = colorPalette[element.index % colorPalette.length];

        // Create highlight overlay using absolute positioning
        const highlight = document.createElement('div');
        highlight.className = 'vibe-indexed-highlight';
        highlight.setAttribute('data-element-index', element.index.toString());
        highlight.style.cssText = `
          position: absolute;
          top: ${rect.top}px;
          left: ${rect.left}px;
          width: ${rect.width}px;
          height: ${rect.height}px;
          border: 3px solid ${colors.border};
          background: ${colors.bg};
          pointer-events: none;
          z-index: 999999;
          box-sizing: border-box;
        `;

        // Create index label using absolute positioning
        const label = document.createElement('div');
        label.className = 'vibe-indexed-label';
        label.setAttribute('data-element-index', element.index.toString());
        label.textContent = element.index.toString();
        label.style.cssText = `
          position: absolute;
          top: ${Math.max(0, rect.top - 28)}px;
          left: ${rect.left}px;
          background: ${colors.label};
          color: white;
          padding: 4px 8px;
          font-size: 14px;
          font-weight: bold;
          border-radius: 4px;
          z-index: 999999;
          pointer-events: none;
          font-family: 'Arial', sans-serif;
          min-width: 24px;
          text-align: center;
          box-shadow: 0 2px 4px rgba(0,0,0,0.4);
          border: 2px solid white;
        `;

        document.body.appendChild(highlight);
        document.body.appendChild(label);
      } catch (e) {
        console.warn('Failed to highlight element:', e);
      }
    });

    // Auto-cleanup after duration
    if (duration > 0) {
      this.highlightTimeout = setTimeout(() => this.cleanupHighlights(), duration);
    }
  }

  /**
   * Get absolute position of element relative to document
   */
  getAbsolutePosition(element) {
    const rect = element.getBoundingClientRect();
    const scrollLeft = window.pageXOffset || document.documentElement.scrollLeft;
    const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
    
    return {
      top: rect.top + scrollTop,
      left: rect.left + scrollLeft,
      width: rect.width,
      height: rect.height
    };
  }

  /**
   * Find DOM element for a given index
   */
  findDomElementForIndex(index) {
    // Try to get from cache first
    if (typeof window !== 'undefined' && window.ElementCache && this.lastCacheId) {
      try {
        const cachedElements = window.ElementCache.get(this.lastCacheId);
        if (cachedElements && cachedElements[index]) {
          return cachedElements[index];
        }
      } catch (e) {
        console.warn('Failed to get element from cache:', e);
      }
    }

    // Fallback: try to find element using data attributes or other markers
    // This is less reliable but better than nothing
    const elementWithIndex = document.querySelector(`[data-vibe-index="${index}"]`);
    if (elementWithIndex) {
      return elementWithIndex;
    }

    // Another fallback: re-extract elements and match by index
    // This is expensive but ensures we get the right element
    try {
      const { domElements } = this.extractInteractiveElements({
        maxElements: 1000
      });
      return domElements[index];
    } catch (e) {
      console.warn('Failed to re-extract elements:', e);
      return null;
    }
  }

  /**
   * Remove all highlighting elements
   */
  cleanupHighlights() {
    const highlights = document.querySelectorAll('.vibe-indexed-highlight, .vibe-indexed-label');
    highlights.forEach(el => el.remove());
    
    // Clear timeout if exists
    if (this.highlightTimeout) {
      clearTimeout(this.highlightTimeout);
      this.highlightTimeout = null;
    }
  }

  /**
   * Check if element is visible
   * Conservative check - only returns false for explicitly hidden elements
   *
   * Note: More comprehensive visibility checks (opacity, off-screen, parent visibility)
   * were tested but removed as they filtered out too many valid interactive elements
   * on modern SPAs. Keeping simple display/visibility check only.
   */
  isElementVisible(element) {
    const style = window.getComputedStyle(element);

    // Only check display and visibility - not opacity or size
    // Many modern UIs use opacity for animations and overlays
    // Size checks can be wrong for elements with text that wraps
    if (style.display === 'none' || style.visibility === 'hidden') {
      return false;
    }

    return true;
  }

  /**
   * Mark selected text within a text node with **>><<** markers
   * When user selects text on the page, this wraps it so LLM can see what was selected
   * @param {Node} textNode - The DOM text node
   * @param {string} text - The trimmed text content
   * @returns {string} Text with selection markers if applicable
   */
  markSelectedText(textNode, text) {
    const selection = window.getSelection();
    if (!selection || selection.rangeCount === 0 || selection.isCollapsed) {
      return text;
    }

    const range = selection.getRangeAt(0);
    if (!range.intersectsNode(textNode)) {
      return text;
    }

    const nodeText = textNode.textContent || '';
    
    // Calculate offsets: use range offsets if this node is the start/end container,
    // otherwise use full node boundaries (0 or length)
    const startOffset = range.startContainer === textNode ? range.startOffset : 0;
    const endOffset = range.endContainer === textNode ? range.endOffset : nodeText.length;

    const selected = nodeText.substring(startOffset, endOffset);
    if (!selected.trim()) {
      return text;
    }
    
    const before = nodeText.substring(0, startOffset);
    const after = nodeText.substring(endOffset);
    return `${before}**>>${selected}<<**${after}`.trim();
  }

  /**
   * Alias for extractContent to maintain backward compatibility with extension
   * The extension expects extractIndexedMarkdown method name
   */
  extractIndexedMarkdown(options = {}) {
    return this.extractContent(options);
  }
}

// Make it available in the global scope for Playwright eval context
// In Playwright's page.evaluate(), code runs in browser context without full module system,
// so we need to ensure MarkdownPageExtractor is globally available after eval()
globalThis.MarkdownPageExtractor = MarkdownPageExtractor;

// ES6 exports for module systems
export default MarkdownPageExtractor;
export { MarkdownPageExtractor };

// Backward compatibility alias
globalThis.IndexedContentExtractor = MarkdownPageExtractor;
