/**
 * Element Cache for Indexed Elements
 * 
 * This module provides a way to cache DOM elements when they are indexed
 * during page content extraction, and retrieve them later for interactions.
 * 
 * The cache is stored on the window object to persist across evaluate calls.
 */

const CACHE_KEY = '__vibeIndexedElementsCache';
const CACHE_EXPIRY_MS = 30000; // 30 seconds

class ElementCache {
  /**
   * Store indexed elements in the cache
   * @param {Array} domElements - Array of DOM elements indexed by position
   * @returns {string} Cache ID for retrieving elements later
   */
  static store(domElements) {
    if (typeof window === 'undefined') {
      throw new Error('ElementCache requires a browser window context');
    }

    // Initialize cache if it doesn't exist
    if (!window[CACHE_KEY]) {
      window[CACHE_KEY] = {
        caches: {},
        nextId: 1
      };
    }

    // Generate unique cache ID
    const cacheId = `cache_${window[CACHE_KEY].nextId++}_${Date.now()}`;
    
    // Store elements with timestamp
    window[CACHE_KEY].caches[cacheId] = {
      elements: domElements,
      timestamp: Date.now(),
      count: domElements.length
    };

    // Clean up old caches
    this.cleanup();

    return cacheId;
  }

  /**
   * Retrieve a specific element by index from cache
   * @param {string} cacheId - Cache ID from store()
   * @param {number} index - Element index
   * @returns {Element|null} The DOM element or null if not found
   */
  static getElement(cacheId, index) {
    if (typeof window === 'undefined' || !window[CACHE_KEY]) {
      return null;
    }

    const cache = window[CACHE_KEY].caches[cacheId];
    if (!cache) {
      console.warn(`ElementCache: Cache ${cacheId} not found`);
      return null;
    }

    // Check if cache is expired
    if (Date.now() - cache.timestamp > CACHE_EXPIRY_MS) {
      console.warn(`ElementCache: Cache ${cacheId} has expired`);
      delete window[CACHE_KEY].caches[cacheId];
      return null;
    }

    // Return element at index
    if (index < 0 || index >= cache.elements.length) {
      console.warn(`ElementCache: Index ${index} out of range for cache ${cacheId}`);
      return null;
    }

    const element = cache.elements[index];
    
    // Verify element is still in DOM
    if (!document.contains(element)) {
      console.warn(`ElementCache: Element at index ${index} is no longer in DOM`);
      return null;
    }

    return element;
  }

  /**
   * Get the most recent cache ID
   * @returns {string|null} The most recent cache ID or null
   */
  static getLatestCacheId() {
    if (typeof window === 'undefined' || !window[CACHE_KEY]) {
      return null;
    }

    const cacheIds = Object.keys(window[CACHE_KEY].caches);
    if (cacheIds.length === 0) {
      return null;
    }

    // Sort by timestamp (embedded in cache ID)
    cacheIds.sort((a, b) => {
      const timeA = parseInt(a.split('_')[2] || '0');
      const timeB = parseInt(b.split('_')[2] || '0');
      return timeB - timeA;
    });

    return cacheIds[0];
  }

  /**
   * Clean up expired caches
   */
  static cleanup() {
    if (typeof window === 'undefined' || !window[CACHE_KEY]) {
      return;
    }

    const now = Date.now();
    const caches = window[CACHE_KEY].caches;
    
    Object.keys(caches).forEach(cacheId => {
      if (now - caches[cacheId].timestamp > CACHE_EXPIRY_MS) {
        delete caches[cacheId];
      }
    });
  }

  /**
   * Clear all caches
   */
  static clearAll() {
    if (typeof window !== 'undefined' && window[CACHE_KEY]) {
      window[CACHE_KEY].caches = {};
    }
  }
}

// Export for use in modules
if (typeof module !== 'undefined' && module.exports) {
  module.exports = ElementCache;
}

// Make available globally in browser context
if (typeof window !== 'undefined') {
  window.ElementCache = ElementCache;
}