// lib/sentry-config.js
// Sentry error tracking configuration for Vibe extension
import * as Sentry from "@sentry/browser";

/**
 * Initialize Sentry for error tracking
 * @param {Object} options - Initialization options
 * @param {string} options.context - Execution context (background, content, ui)
 * @param {string} options.dsn - Sentry DSN (optional, uses env var if not provided)
 * @param {string} options.release - Release version (optional, uses manifest if not provided)
 * @param {string} options.environment - Environment (production, development, etc.)
 */
export function initSentry(options = {}) {
  const {
    context = 'unknown',
    dsn = process.env.SENTRY_DSN || '',
    release = null,
    environment = process.env.SENTRY_ENVIRONMENT || (process.env.NODE_ENV === 'production' ? 'production' : 'development')
  } = options;

  // Skip initialization if no DSN provided
  if (!dsn || dsn.trim() === '') {
    console.log('[Sentry] Skipping initialization - no DSN provided');
    return false;
  }

  // Content scripts cannot initialize Sentry directly
  // They must send errors to background script instead
  if (context === 'content') {
    console.log('[Sentry] Skipping initialization in content script - use message passing instead');
    return false;
  }

  try {
    Sentry.init({
      dsn,
      environment,
      release: release || `vibe-extension@${getManifestVersion()}`,

      // Configure integrations
      integrations: [
        Sentry.browserTracingIntegration({
          // Disable automatic instrumentation that might not work in extensions
          tracePropagationTargets: [],
        }),
        Sentry.replayIntegration({
          // Disable session replay for privacy
          maskAllText: true,
          blockAllMedia: true,
        }),
      ],

      // Performance monitoring
      tracesSampleRate: environment === 'production' ? 0.1 : 1.0,

      // Session replay - disabled for privacy by default
      replaysSessionSampleRate: 0.0,
      replaysOnErrorSampleRate: 0.0,

      // Add context tags
      initialScope: {
        tags: {
          context,
          extension: 'vibe',
        },
      },

      // Filter sensitive data before sending
      beforeSend(event, hint) {
        // Remove sensitive data from breadcrumbs and context
        if (event.breadcrumbs) {
          event.breadcrumbs = event.breadcrumbs.map(breadcrumb => {
            return sanitizeBreadcrumb(breadcrumb);
          });
        }

        // Sanitize error messages and stack traces
        if (event.exception?.values) {
          event.exception.values = event.exception.values.map(exception => {
            if (exception.value) {
              exception.value = sanitizeErrorMessage(exception.value);
            }
            return exception;
          });
        }

        // Remove request data that might contain API keys
        if (event.request) {
          delete event.request.cookies;
          delete event.request.headers;

          // Sanitize URL parameters
          if (event.request.url) {
            event.request.url = sanitizeUrl(event.request.url);
          }
        }

        // Remove user data
        delete event.user;

        return event;
      },
    });

    console.log(`[Sentry] Initialized for ${context} context in ${environment} environment`);
    return true;
  } catch (error) {
    console.error('[Sentry] Failed to initialize:', error);
    return false;
  }
}

/**
 * Get manifest version for release tracking
 */
function getManifestVersion() {
  try {
    if (typeof chrome !== 'undefined' && chrome.runtime?.getManifest) {
      const manifest = chrome.runtime.getManifest();
      return manifest.version_name || manifest.version || 'unknown';
    }
  } catch (error) {
    console.warn('[Sentry] Could not get manifest version:', error);
  }
  return 'unknown';
}

/**
 * Sanitize breadcrumb data to remove sensitive information
 */
function sanitizeBreadcrumb(breadcrumb) {
  const sanitized = { ...breadcrumb };

  // Remove sensitive data from breadcrumb data
  if (sanitized.data) {
    const data = { ...sanitized.data };

    // Remove API keys and tokens
    const sensitiveKeys = ['apiKey', 'api_key', 'token', 'password', 'secret', 'authorization'];
    sensitiveKeys.forEach(key => {
      if (data[key]) {
        data[key] = '[REDACTED]';
      }
    });

    // Sanitize URLs
    if (data.url) {
      data.url = sanitizeUrl(data.url);
    }

    sanitized.data = data;
  }

  return sanitized;
}

/**
 * Sanitize error messages to remove sensitive data
 */
function sanitizeErrorMessage(message) {
  if (typeof message !== 'string') return message;

  // Remove API key patterns (common formats)
  let sanitized = message;

  // OpenAI API keys: sk-...
  sanitized = sanitized.replace(/sk-[a-zA-Z0-9]{32,}/g, 'sk-[REDACTED]');

  // Generic API key patterns
  sanitized = sanitized.replace(/[a-zA-Z0-9_-]{32,}/g, (match) => {
    // Only redact if it looks like a key (all alphanumeric, long enough)
    if (match.length >= 32 && /^[a-zA-Z0-9_-]+$/.test(match)) {
      return '[REDACTED]';
    }
    return match;
  });

  // Remove authorization headers
  sanitized = sanitized.replace(/authorization:\s*[^\s,}]+/gi, 'authorization: [REDACTED]');
  sanitized = sanitized.replace(/bearer\s+[^\s,}]+/gi, 'bearer [REDACTED]');

  return sanitized;
}

/**
 * Sanitize URLs to remove sensitive query parameters
 */
function sanitizeUrl(url) {
  try {
    const urlObj = new URL(url);
    const sensitiveParams = ['api_key', 'apikey', 'token', 'password', 'secret', 'key'];

    sensitiveParams.forEach(param => {
      if (urlObj.searchParams.has(param)) {
        urlObj.searchParams.set(param, '[REDACTED]');
      }
    });

    return urlObj.toString();
  } catch (error) {
    // If URL parsing fails, return as-is
    return url;
  }
}

/**
 * Capture an exception with additional context
 * @param {Error} error - The error to capture
 * @param {Object} context - Additional context to attach
 */
export function captureException(error, context = {}) {
  // If running in content script, send error to background script
  if (context.context === 'content' && typeof chrome !== 'undefined' && chrome.runtime?.sendMessage) {
    try {
      chrome.runtime.sendMessage({
        type: 'SENTRY_CAPTURE_EXCEPTION',
        payload: {
          message: error?.message || String(error),
          stack: error?.stack,
          context: sanitizeContext(context)
        }
      }).catch(err => {
        console.error('[Sentry] Failed to send error to background:', err);
      });
      return;
    } catch (err) {
      console.error('[Sentry] Failed to send error to background:', err);
      return;
    }
  }

  // Sanitize context to remove sensitive data
  const sanitizedContext = sanitizeContext(context);

  Sentry.captureException(error, {
    contexts: {
      vibe: sanitizedContext,
    },
  });
}

/**
 * Sanitize context to remove sensitive data
 */
function sanitizeContext(context) {
  const sanitizedContext = { ...context };
  
  // Remove sensitive fields
  const sensitiveFields = ['apiKey', 'api_key', 'password', 'token', 'secret'];
  sensitiveFields.forEach(field => {
    if (sanitizedContext[field]) {
      sanitizedContext[field] = '[REDACTED]';
    }
  });
  
  return sanitizedContext;
}

/**
 * Capture a message with optional level
 * @param {string} message - The message to capture
 * @param {string} level - Severity level (info, warning, error)
 */
export function captureMessage(message, level = 'info') {
  const sanitized = sanitizeErrorMessage(message);
  Sentry.captureMessage(sanitized, level);
}

/**
 * Add breadcrumb for debugging context
 * @param {Object} breadcrumb - Breadcrumb data
 */
export function addBreadcrumb(breadcrumb) {
  const sanitized = sanitizeBreadcrumb(breadcrumb);
  Sentry.addBreadcrumb(sanitized);
}

/**
 * Set user context (use sparingly and without PII)
 * @param {Object} user - User data (should not contain PII)
 */
export function setUser(user) {
  // Only allow non-PII user data
  const safeUser = {
    id: user.id ? `user_${hashString(user.id)}` : undefined,
  };
  Sentry.setUser(safeUser);
}

/**
 * Set tags for filtering and grouping
 * @param {Object} tags - Tags to set
 */
export function setTags(tags) {
  Object.entries(tags).forEach(([key, value]) => {
    Sentry.setTag(key, value);
  });
}

/**
 * Simple hash function for anonymizing IDs
 */
function hashString(str) {
  let hash = 0;
  for (let i = 0; i < str.length; i++) {
    const char = str.charCodeAt(i);
    hash = ((hash << 5) - hash) + char;
    hash = hash & hash; // Convert to 32-bit integer
  }
  return Math.abs(hash).toString(36);
}

// Export Sentry for advanced usage
export { Sentry };
