import { isNaturalLanguageQuery, extractQueryFromUrl } from './query_utils.js';

describe('Query Detection Functions', () => {
  describe('isNaturalLanguageQuery', () => {
    test('should detect natural language travel queries', () => {
      expect(isNaturalLanguageQuery('book a flight to New York')).toBe(true);
      expect(isNaturalLanguageQuery('find a hotel in Paris')).toBe(true);
      expect(isNaturalLanguageQuery('search for restaurants near me')).toBe(true);
      expect(isNaturalLanguageQuery('book a flight from San Francisco to Tokyo')).toBe(true);
    });

    test('should detect general natural language queries', () => {
      expect(isNaturalLanguageQuery('what is the weather today')).toBe(true);
      expect(isNaturalLanguageQuery('how to cook pasta')).toBe(true);
      expect(isNaturalLanguageQuery('find the best laptop under 1000')).toBe(true);
      expect(isNaturalLanguageQuery('help me with my homework')).toBe(true);
    });

    test('should detect search URLs as natural language', () => {
      expect(isNaturalLanguageQuery('https://www.google.com/search?q=book+a+flight+to+New+York')).toBe(true);
      expect(isNaturalLanguageQuery('https://www.google.com/search?q=find+restaurants+near+me')).toBe(true);
    });

    test('should NOT detect regular URLs as natural language', () => {
      expect(isNaturalLanguageQuery('https://www.google.com')).toBe(false);
      expect(isNaturalLanguageQuery('https://github.com/user/repo')).toBe(false);
      expect(isNaturalLanguageQuery('https://stackoverflow.com/questions/123')).toBe(false);
      expect(isNaturalLanguageQuery('www.example.com')).toBe(false);
    });

    test('should NOT detect chrome URLs as natural language', () => {
      expect(isNaturalLanguageQuery('chrome://settings')).toBe(false);
      expect(isNaturalLanguageQuery('chrome-extension://abcdef/popup.html')).toBe(false);
      expect(isNaturalLanguageQuery('about:blank')).toBe(false);
      expect(isNaturalLanguageQuery('file:///path/to/file.html')).toBe(false);
    });

    test('should NOT detect short or empty queries', () => {
      expect(isNaturalLanguageQuery('')).toBe(false);
      expect(isNaturalLanguageQuery('ab')).toBe(false);
      expect(isNaturalLanguageQuery('a')).toBe(false);
    });

    test('should NOT detect domain-like patterns', () => {
      expect(isNaturalLanguageQuery('example.com')).toBe(false);
      expect(isNaturalLanguageQuery('sub.domain.example.co.uk')).toBe(false);
      expect(isNaturalLanguageQuery('localhost:3000')).toBe(false);
    });

    test('should handle encoded URLs', () => {
      expect(isNaturalLanguageQuery('book%20a%20flight%20to%20New%20York')).toBe(true);
      expect(isNaturalLanguageQuery('find%20restaurants%20near%20me')).toBe(true);
    });
  });

  describe('extractQueryFromUrl', () => {
    test('should extract query from search URLs', () => {
      expect(extractQueryFromUrl('https://www.google.com/search?q=book+a+flight+to+New+York'))
        .toBe('book a flight to New York');
      expect(extractQueryFromUrl('https://www.google.com/search?q=find+restaurants+near+me'))
        .toBe('find restaurants near me');
    });

    test('should extract query from direct input', () => {
      expect(extractQueryFromUrl('book a flight to New York')).toBe('book a flight to New York');
      expect(extractQueryFromUrl('find restaurants near me')).toBe('find restaurants near me');
    });

    test('should handle encoded queries', () => {
      expect(extractQueryFromUrl('book%20a%20flight%20to%20New%20York')).toBe('book a flight to New York');
      expect(extractQueryFromUrl('find%20restaurants%20near%20me')).toBe('find restaurants near me');
    });

    test('should handle URLs with protocol', () => {
      expect(extractQueryFromUrl('https://book a flight to New York')).toBe('book a flight to New York');
      expect(extractQueryFromUrl('http://find restaurants near me')).toBe('find restaurants near me');
    });

    test('should handle + symbols in queries', () => {
      expect(extractQueryFromUrl('book+a+flight+to+New+York')).toBe('book a flight to New York');
      expect(extractQueryFromUrl('find+restaurants+near+me')).toBe('find restaurants near me');
    });

    test('should trim whitespace', () => {
      expect(extractQueryFromUrl('  book a flight to New York  ')).toBe('book a flight to New York');
      expect(extractQueryFromUrl('find restaurants near me/')).toBe('find restaurants near me');
    });
  });
});