/**
 * Detects if a URL represents a natural language query
 */
export function isNaturalLanguageQuery(url) {
  // Skip chrome:// and extension URLs
  if (url.startsWith("chrome://") || url.startsWith("chrome-extension://")) {
    return false;
  }

  // Skip file:// URLs
  if (url.startsWith("file://")) {
    return false;
  }

  // Skip about: URLs
  if (url.startsWith("about:")) {
    return false;
  }

  let query = "";
  // Handle search URLs (when user types in address bar, Chrome creates search URLs)
  if (url.startsWith("https://www.google.com/search?q=") || 
      url.startsWith("http://www.google.com/search?q=") ||
      url.includes("/search?q=")) {
    const urlObj = new URL(url);
    query = urlObj.searchParams.get("q") || "";
  } else if (url.startsWith("http://") || url.startsWith("https://")) {
    // If it's an http/https URL but contains spaces, it's likely a natural language query
    const decodedUrl = decodeURIComponent(url);
    if (decodedUrl.includes(" ")) {
      query = decodedUrl.replace(/^[a-z]+:\/\//, "").replace(/\/$/, "").replace(/\+/g, " ").trim();
    } else {
      // If it's a valid URL without spaces, it's not a natural language query
      return false;
    }
  } else {
    // Direct query (no protocol)
    query = decodeURIComponent(url);
  }

  // Clean the query
  query = query.toLowerCase().trim();

  // Skip empty queries
  if (!query || query.length < 3) {
    return false;
  }

  // Skip obvious URLs without protocol (domain-like patterns)
  if (/^[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}(\/.*)?$/.test(query)) {
    return false;
  }

  // Enhanced natural language detection patterns
  const naturalLanguagePatterns = [
    // Travel & booking
    /book.*flight/,
    /find.*hotel/,
    /search.*restaurant/,
    /order.*food/,
    /buy.*ticket/,
    /plan.*trip/,
    /travel.*to/,

    // Actions with common patterns
    /book.*from.*to/,
    /flight.*from.*to/,

    // General patterns - multi-word queries with spaces
    /\b(get|find|search|look|show|tell|help|book|buy|order|plan|schedule|create|make)\b.*\s+/
  ];

  // Check for multiple words (likely natural language)
  const hasMultipleWords = query.split(/\s+/).length >= 2;

  return naturalLanguagePatterns.some(pattern => pattern.test(query)) || 
         (hasMultipleWords && query.length > 10);
}

/**
 * Extracts the natural language query from a URL
 */
export function extractQueryFromUrl(url) {
  let query = "";

  try {
    const urlObj = new URL(url);
    // Prioritize extracting from 'q' parameter if it exists
    if (urlObj.searchParams.has("q")) {
      query = urlObj.searchParams.get("q") || "";
      return decodeURIComponent(query).trim();
    }
  } catch (error) {
    // URL might be malformed or not a standard URL, proceed to treat as direct query
  }

  // If not a search URL with 'q' parameter, assume the entire path (after decoding and cleaning) is the query
  // This handles cases where Chrome might prepend http/https to a natural language input
  query = url.replace(/^[a-z]+:\/\//, ""); // Remove protocol if present
  query = decodeURIComponent(query);
  query = query.replace(/\/$/, ""); // Remove trailing slash
  query = query.replace(/\+/g, " "); // Replace + with spaces

  return query.trim();
}