/**
 * XML Tool Call Parser
 *
 * Character-by-character streaming parser for extracting tool calls from LLM responses.
 * Adapted from Kilo Code's AssistantMessageParser.ts for use with models that don't
 * support native function calling (like Gemini Nano).
 *
 * Expected format:
 * <tool_name>
 * <param1>value1</param1>
 * <param2>value2</param2>
 * </tool_name>
 */

export interface ToolCall {
  id: string;
  name: string;
  args: Record<string, string>;
}

interface ToolUseBlock {
  type: 'tool_use';
  id: string;
  name: string;
  params: Record<string, string>;
  partial: boolean;
}

interface LangChainTool {
  name: string;
  schema?: {
    shape?: Record<string, unknown>;
    _def?: {
      shape: () => Record<string, unknown>;
    };
  };
  parameters?: {
    properties?: Record<string, unknown>;
  };
}

export class XmlToolParser {
  private toolNames: string[];
  private paramNames: string[];
  private accumulator: string = '';
  private contentBlocks: ToolUseBlock[] = [];
  private currentToolUse: ToolUseBlock | null = null;
  private currentParamName: string | null = null;
  private paramValueStartIndex: number = 0;
  private toolUseStartIndex: number = 0;

  constructor(toolNames: string[] = [], paramNames: string[] = []) {
    this.toolNames = toolNames;
    this.paramNames = paramNames;
    this.reset();
  }

  reset(): void {
    this.accumulator = '';
    this.contentBlocks = [];
    this.currentToolUse = null;
    this.currentParamName = null;
    this.paramValueStartIndex = 0;
    this.toolUseStartIndex = 0;
  }

  setToolNames(toolNames: string[]): void {
    this.toolNames = toolNames;
  }

  setParamNames(paramNames: string[]): void {
    this.paramNames = paramNames;
  }

  processChunk(chunk: string): ToolUseBlock[] {
    for (const char of chunk) {
      this.accumulator += char;

      // State 1: Inside a parameter value
      if (this.currentToolUse && this.currentParamName) {
        const closingTag = `</${this.currentParamName}>`;

        if (this.accumulator.endsWith(closingTag)) {
          const paramValue = this.accumulator.slice(
            this.paramValueStartIndex,
            -closingTag.length
          );

          // Trim but preserve internal whitespace for content params
          this.currentToolUse.params[this.currentParamName] =
            this.currentParamName === 'content'
              ? paramValue.replace(/^\n/, '').replace(/\n$/, '')
              : paramValue.trim();

          this.currentParamName = null;
          continue;
        }
        continue;
      }

      // State 2: Inside a tool but not in a parameter
      if (this.currentToolUse) {
        const toolCloseTag = `</${this.currentToolUse.name}>`;

        if (this.accumulator.endsWith(toolCloseTag)) {
          this.currentToolUse.partial = false;
          this.currentToolUse = null;
          continue;
        }

        // Check for parameter opening tags
        for (const paramName of this.paramNames) {
          const paramOpenTag = `<${paramName}>`;
          if (this.accumulator.endsWith(paramOpenTag)) {
            this.currentParamName = paramName;
            this.paramValueStartIndex = this.accumulator.length;
            break;
          }
        }
        continue;
      }

      // State 3: Not in a tool - check for tool opening
      for (const toolName of this.toolNames) {
        const toolOpenTag = `<${toolName}>`;
        if (this.accumulator.endsWith(toolOpenTag)) {
          this.currentToolUse = {
            type: 'tool_use',
            name: toolName,
            params: {},
            partial: true,
            id: `nano_${Date.now()}_${Math.random().toString(36).slice(2, 9)}`
          };
          this.toolUseStartIndex = this.accumulator.length;
          this.contentBlocks.push(this.currentToolUse);
          break;
        }
      }
    }

    return this.contentBlocks;
  }

  getToolCalls(): ToolCall[] {
    return this.contentBlocks
      .filter(block => block.type === 'tool_use' && !block.partial)
      .map(block => ({
        id: block.id,
        name: block.name,
        args: block.params
      }));
  }

  getAllToolCalls(): ToolUseBlock[] {
    return this.contentBlocks.filter(block => block.type === 'tool_use');
  }

  hasPartialToolCall(): boolean {
    return this.currentToolUse !== null;
  }

  getTextContent(): string {
    let text = this.accumulator;

    for (const block of this.contentBlocks) {
      if (!block.partial) {
        let toolString = `<${block.name}>`;
        for (const [param, value] of Object.entries(block.params)) {
          toolString += `<${param}>${value}</${param}>`;
        }
        toolString += `</${block.name}>`;
        text = text.replace(new RegExp(escapeRegex(toolString), 'g'), '');
      }
    }

    return text.trim();
  }
}

function escapeRegex(string: string): string {
  return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}

export function extractParamNames(tools: LangChainTool[]): string[] {
  const params = new Set<string>();

  for (const tool of tools) {
    if (tool.schema?.shape) {
      Object.keys(tool.schema.shape).forEach(k => params.add(k));
    }
    if (tool.schema?._def?.shape) {
      const shape = tool.schema._def.shape();
      Object.keys(shape).forEach(k => params.add(k));
    }
    if (tool.parameters?.properties) {
      Object.keys(tool.parameters.properties).forEach(k => params.add(k));
    }
  }

  return Array.from(params);
}

export default XmlToolParser;
