/**
 * Models.dev Integration
 * 
 * Fetches model metadata from https://models.dev/api.json
 * Provides model capabilities, pricing, and configuration for any provider.
 * 
 * This is metadata-only - actual LLM calls use LangChain with the provider's API.
 * Most providers use OpenAI-compatible APIs, so ChatOpenAI works with them.
 */

export interface ModelCost {
  input: number;      // Cost per 1M input tokens
  output: number;     // Cost per 1M output tokens
  cache_read?: number;
  cache_write?: number;
}

export interface ModelLimit {
  context: number;    // Max context window
  output: number;     // Max output tokens
}

export interface ModelModalities {
  input: string[];    // ["text", "image", "audio", "video", "pdf"]
  output: string[];
}

export interface ModelInfo {
  id: string;
  name: string;
  family?: string;
  release_date: string;
  attachment: boolean;      // Supports file attachments
  reasoning: boolean;       // Has reasoning/thinking mode
  temperature: boolean;     // Supports temperature control
  tool_call: boolean;       // Supports function calling
  cost?: ModelCost;
  limit: ModelLimit;
  modalities?: ModelModalities;
  status?: 'alpha' | 'beta' | 'deprecated';
  options?: Record<string, any>;
  headers?: Record<string, string>;
}

export interface ProviderInfo {
  id: string;
  name: string;
  api?: string;             // Base URL for API
  env: string[];            // Environment variables for API key
  npm?: string;             // AI SDK package (for reference)
  models: Record<string, ModelInfo>;
}

export type ModelsDevData = Record<string, ProviderInfo>;

const MODELS_DEV_URL = 'https://models.dev/api.json';
const CACHE_KEY = 'vibe_models_dev_cache';
const CACHE_DURATION = 60 * 60 * 1000; // 1 hour

/**
 * Models.dev client for fetching model metadata
 */
export class ModelsDevClient {
  private cache: ModelsDevData | null = null;
  private cacheTime: number = 0;

  /**
   * Fetch models.dev data with caching
   */
  async getModels(): Promise<ModelsDevData> {
    // Check memory cache
    if (this.cache && Date.now() - this.cacheTime < CACHE_DURATION) {
      return this.cache;
    }

    // Check storage cache
    if (typeof chrome !== 'undefined' && chrome.storage) {
      try {
        const result = await chrome.storage.local.get(CACHE_KEY);
        const cached = result[CACHE_KEY];
        if (cached && Date.now() - cached.time < CACHE_DURATION) {
          this.cache = cached.data;
          this.cacheTime = cached.time;
          return this.cache as ModelsDevData;
        }
      } catch {
        // Ignore storage errors
      }
    }

    // Fetch fresh data
    try {
      const response = await fetch(MODELS_DEV_URL, {
        signal: AbortSignal.timeout(10000),
      });
      
      if (!response.ok) {
        throw new Error(`Failed to fetch models.dev: ${response.status}`);
      }

      this.cache = await response.json();
      this.cacheTime = Date.now();

      // Save to storage cache
      if (typeof chrome !== 'undefined' && chrome.storage) {
        chrome.storage.local.set({
          [CACHE_KEY]: { data: this.cache, time: this.cacheTime }
        }).catch(() => {});
      }

      return this.cache!;
    } catch (error) {
      // Return cached data if available, even if expired
      if (this.cache) {
        console.warn('[ModelsDevClient] Using stale cache due to fetch error:', error);
        return this.cache;
      }
      throw error;
    }
  }

  /**
   * Get provider info by ID
   */
  async getProvider(providerId: string): Promise<ProviderInfo | null> {
    const data = await this.getModels();
    return data[providerId] || null;
  }

  /**
   * Get model info by provider and model ID
   */
  async getModel(providerId: string, modelId: string): Promise<ModelInfo | null> {
    const provider = await this.getProvider(providerId);
    return provider?.models[modelId] || null;
  }

  /**
   * Get all providers
   */
  async getProviders(): Promise<ProviderInfo[]> {
    const data = await this.getModels();
    return Object.values(data);
  }

  /**
   * Search for models by capability
   */
  async searchModels(options: {
    toolCall?: boolean;
    reasoning?: boolean;
    vision?: boolean;
    minContext?: number;
    maxCostPerMillion?: number;
  }): Promise<Array<{ provider: ProviderInfo; model: ModelInfo }>> {
    const data = await this.getModels();
    const results: Array<{ provider: ProviderInfo; model: ModelInfo }> = [];

    for (const provider of Object.values(data)) {
      for (const model of Object.values(provider.models)) {
        let matches = true;

        if (options.toolCall !== undefined && model.tool_call !== options.toolCall) {
          matches = false;
        }
        if (options.reasoning !== undefined && model.reasoning !== options.reasoning) {
          matches = false;
        }
        if (options.vision && !model.modalities?.input?.includes('image')) {
          matches = false;
        }
        if (options.minContext && model.limit.context < options.minContext) {
          matches = false;
        }
        if (options.maxCostPerMillion && model.cost && model.cost.input > options.maxCostPerMillion) {
          matches = false;
        }

        if (matches) {
          results.push({ provider, model });
        }
      }
    }

    return results;
  }
}

/**
 * Map models.dev provider to LangChain configuration
 * Most providers use OpenAI-compatible APIs
 */
export function mapToLangChainConfig(provider: ProviderInfo, modelId: string): {
  baseURL: string;
  model: string;
  providerType: 'openai' | 'anthropic' | 'google' | 'azure' | 'custom';
} {
  // Determine provider type based on npm package or provider ID
  let providerType: 'openai' | 'anthropic' | 'google' | 'azure' | 'custom' = 'openai';
  
  if (provider.id === 'anthropic' || provider.npm === '@ai-sdk/anthropic') {
    providerType = 'anthropic';
  } else if (provider.id === 'google' || provider.npm === '@ai-sdk/google') {
    providerType = 'google';
  } else if (provider.id === 'azure' || provider.npm === '@ai-sdk/azure') {
    providerType = 'azure';
  } else if (provider.npm?.includes('openai-compatible') || provider.api) {
    // Most providers use OpenAI-compatible API
    providerType = 'openai';
  }

  return {
    baseURL: provider.api || '',
    model: modelId,
    providerType,
  };
}

/**
 * Get the environment variable name for a provider's API key
 */
export function getApiKeyEnvVar(provider: ProviderInfo): string | null {
  return provider.env[0] || null;
}

/**
 * Format cost for display (per 1M tokens)
 */
export function formatCost(cost: ModelCost | undefined): string {
  if (!cost) return 'Free';
  if (cost.input === 0 && cost.output === 0) return 'Free';
  return `$${cost.input}/${cost.output} per 1M`;
}

/**
 * Format context limit for display
 */
export function formatContextLimit(limit: ModelLimit): string {
  if (limit.context >= 1000000) {
    return `${(limit.context / 1000000).toFixed(1)}M`;
  }
  if (limit.context >= 1000) {
    return `${Math.round(limit.context / 1000)}K`;
  }
  return limit.context.toString();
}

// Singleton instance
export const modelsDevClient = new ModelsDevClient();

/**
 * Quick lookup for common providers that are OpenAI-compatible
 * These can be used with ChatOpenAI by setting baseURL
 */
export const OPENAI_COMPATIBLE_PROVIDERS = [
  'openrouter',
  'together',
  'groq',
  'deepinfra',
  'fireworks',
  'perplexity',
  'mistral',
  'cerebras',
  'ollama',
  'ollama-cloud',
  'moonshot',
  'moonshotai',
  'alibaba',
  'xai',
  'sambanova',
  'cohere',
  // GitHub Copilot uses OpenAI-compatible API
  'github-copilot',
];

/**
 * Providers that need special handling (not OpenAI-compatible)
 */
export const SPECIAL_PROVIDERS = {
  'anthropic': 'ChatAnthropic',
  'google': 'ChatGoogleGenerativeAI',
  'azure': 'AzureChatOpenAI',
  'amazon-bedrock': 'ChatBedrock',
};
