/**
 * GitHub Copilot Provider
 * 
 * Implements OAuth device flow authentication for GitHub Copilot.
 * Compatible with OpenCode's opencode-copilot-auth plugin architecture.
 * 
 * Usage:
 *   const auth = new GithubCopilotAuth();
 *   const { url, userCode, callback } = await auth.initiateDeviceFlow();
 *   // Show user the url and userCode, wait for them to authorize
 *   const result = await callback();
 *   // Use result.refresh token for API calls
 */

// GitHub Copilot OAuth constants (matching OpenCode's plugin)
const CLIENT_ID = "Iv1.b507a08c87ecfe98";

const COPILOT_HEADERS = {
  "User-Agent": "GitHubCopilotChat/0.32.4",
  "Editor-Version": "vscode/1.105.1",
  "Editor-Plugin-Version": "copilot-chat/0.32.4",
  "Copilot-Integration-Id": "vscode-chat",
};

// Response API alternate input types for agent detection
const RESPONSES_API_ALTERNATE_INPUT_TYPES = [
  "file_search_call",
  "computer_call",
  "computer_call_output",
  "web_search_call",
  "function_call",
  "function_call_output",
  "image_generation_call",
  "code_interpreter_call",
  "local_shell_call",
  "local_shell_call_output",
  "mcp_list_tools",
  "mcp_approval_request",
  "mcp_approval_response",
  "mcp_call",
  "reasoning",
];

export interface GithubCopilotOAuthState {
  type: 'oauth';
  refresh: string;      // GitHub OAuth token (long-lived)
  access: string;       // Copilot API token (short-lived)
  expires: number;      // Expiry timestamp in ms
  enterpriseUrl?: string; // For GitHub Enterprise
}

export interface DeviceFlowResult {
  url: string;
  userCode: string;
  callback: () => Promise<AuthResult>;
}

export type AuthResult = 
  | { type: 'success'; refresh: string; access: string; expires: number; enterpriseUrl?: string }
  | { type: 'failed' };

function normalizeDomain(url: string): string {
  return url.replace(/^https?:\/\//, "").replace(/\/$/, "");
}

function getUrls(domain: string) {
  return {
    DEVICE_CODE_URL: `https://${domain}/login/device/code`,
    ACCESS_TOKEN_URL: `https://${domain}/login/oauth/access_token`,
    COPILOT_API_KEY_URL: `https://api.${domain}/copilot_internal/v2/token`,
  };
}

/**
 * GitHub Copilot Authentication Handler
 * Implements OAuth device flow for GitHub Copilot
 */
export class GithubCopilotAuth {
  private enterpriseUrl?: string;

  constructor(enterpriseUrl?: string) {
    this.enterpriseUrl = enterpriseUrl;
  }

  /**
   * Get the domain for API calls
   */
  private getDomain(): string {
    return this.enterpriseUrl ? normalizeDomain(this.enterpriseUrl) : "github.com";
  }

  /**
   * Get the base URL for Copilot API
   */
  getBaseURL(): string {
    return this.enterpriseUrl
      ? `https://copilot-api.${normalizeDomain(this.enterpriseUrl)}`
      : "https://api.githubcopilot.com";
  }

  /**
   * Initiate OAuth device flow
   * Returns URL for user to visit and callback to poll for auth
   */
  async initiateDeviceFlow(): Promise<DeviceFlowResult> {
    const domain = this.getDomain();
    const urls = getUrls(domain);

    const deviceResponse = await fetch(urls.DEVICE_CODE_URL, {
      method: "POST",
      headers: {
        Accept: "application/json",
        "Content-Type": "application/json",
        "User-Agent": "GitHubCopilotChat/0.35.0",
      },
      body: JSON.stringify({
        client_id: CLIENT_ID,
        scope: "read:user",
      }),
    });

    if (!deviceResponse.ok) {
      throw new Error("Failed to initiate device authorization");
    }

    const deviceData = await deviceResponse.json();

    return {
      url: deviceData.verification_uri,
      userCode: deviceData.user_code,
      callback: async (): Promise<AuthResult> => {
        // Poll for authorization
        while (true) {
          const response = await fetch(urls.ACCESS_TOKEN_URL, {
            method: "POST",
            headers: {
              Accept: "application/json",
              "Content-Type": "application/json",
              "User-Agent": "GitHubCopilotChat/0.35.0",
            },
            body: JSON.stringify({
              client_id: CLIENT_ID,
              device_code: deviceData.device_code,
              grant_type: "urn:ietf:params:oauth:grant-type:device_code",
            }),
          });

          if (!response.ok) return { type: "failed" };

          const data = await response.json();

          if (data.access_token) {
            return {
              type: "success",
              refresh: data.access_token,
              access: "",
              expires: 0,
              ...(this.enterpriseUrl && { enterpriseUrl: this.enterpriseUrl }),
            };
          }

          if (data.error === "authorization_pending") {
            await new Promise((resolve) =>
              setTimeout(resolve, deviceData.interval * 1000)
            );
            continue;
          }

          if (data.error) return { type: "failed" };

          await new Promise((resolve) =>
            setTimeout(resolve, deviceData.interval * 1000)
          );
        }
      },
    };
  }

  /**
   * Refresh the Copilot API token using the OAuth refresh token
   */
  async refreshToken(auth: GithubCopilotOAuthState): Promise<GithubCopilotOAuthState> {
    const domain = auth.enterpriseUrl ? normalizeDomain(auth.enterpriseUrl) : "github.com";
    const urls = getUrls(domain);

    const response = await fetch(urls.COPILOT_API_KEY_URL, {
      headers: {
        Accept: "application/json",
        Authorization: `Bearer ${auth.refresh}`,
        ...COPILOT_HEADERS,
      },
    });

    if (!response.ok) {
      throw new Error(`Token refresh failed: ${response.status}`);
    }

    const tokenData = await response.json();

    return {
      ...auth,
      access: tokenData.token,
      expires: tokenData.expires_at * 1000,
    };
  }
}

/**
 * Create a custom fetch function for GitHub Copilot
 * Handles automatic token refresh and proper headers
 */
export function createCopilotFetch(
  getAuth: () => Promise<GithubCopilotOAuthState | null>,
  saveAuth: (auth: GithubCopilotOAuthState) => Promise<void>
): typeof fetch {
  const copilotAuth = new GithubCopilotAuth();

  return async (input: RequestInfo | URL, init?: RequestInit): Promise<Response> => {
    let auth = await getAuth();
    
    if (!auth || auth.type !== 'oauth') {
      throw new Error("Not logged in to GitHub Copilot. Please connect in Settings.");
    }

    if (!auth.refresh) {
      throw new Error("GitHub Copilot refresh token missing. Please reconnect in Settings.");
    }

    // Refresh token if expired
    if (!auth.access || auth.expires < Date.now()) {
      auth = await copilotAuth.refreshToken(auth);
      await saveAuth(auth);
    }

    // Detect if this is an agent call or vision request
    let isAgentCall = false;
    let isVisionRequest = false;

    try {
      const body = typeof init?.body === "string" ? JSON.parse(init.body) : init?.body;
      
      if (body?.messages) {
        isAgentCall = body.messages.some(
          (msg: any) => msg.role && ["tool", "assistant"].includes(msg.role)
        );
        isVisionRequest = body.messages.some(
          (msg: any) =>
            Array.isArray(msg.content) &&
            msg.content.some((part: any) => part.type === "image_url")
        );
      }

      if (body?.input) {
        const lastInput = body.input[body.input.length - 1];
        const isAssistant = lastInput?.role === "assistant";
        const hasAgentType = lastInput?.type
          ? RESPONSES_API_ALTERNATE_INPUT_TYPES.includes(lastInput.type)
          : false;
        isAgentCall = isAssistant || hasAgentType;
        isVisionRequest =
          Array.isArray(lastInput?.content) &&
          lastInput.content.some((part: any) => part.type === "input_image");
      }
    } catch {
      // Ignore parse errors
    }

    // Build headers
    const headers: Record<string, string> = {
      ...(init?.headers as Record<string, string> || {}),
      ...COPILOT_HEADERS,
      Authorization: `Bearer ${auth.access}`,
      "Openai-Intent": "conversation-edits",
      "X-Initiator": isAgentCall ? "agent" : "user",
    };

    if (isVisionRequest) {
      headers["Copilot-Vision-Request"] = "true";
    }

    // Remove conflicting headers (handle both cases for case-sensitivity)
    delete headers["x-api-key"];
    delete headers["authorization"];
    delete headers["Authorization"];

    return fetch(input, {
      ...init,
      headers,
    });
  };
}

/**
 * Storage key for GitHub Copilot OAuth
 */
export const COPILOT_AUTH_STORAGE_KEY = "vibe_github_copilot_oauth";

/**
 * Chrome storage helpers for GitHub Copilot auth
 */
export const CopilotStorage = {
  async get(): Promise<GithubCopilotOAuthState | null> {
    if (typeof chrome === 'undefined' || !chrome.storage) {
      return null;
    }
    const result = await chrome.storage.local.get(COPILOT_AUTH_STORAGE_KEY);
    const storedValue = result[COPILOT_AUTH_STORAGE_KEY];
    if (!storedValue) return null;
    try {
      return JSON.parse(storedValue);
    } catch {
      return null;
    }
  },

  async set(auth: GithubCopilotOAuthState): Promise<void> {
    if (typeof chrome === 'undefined' || !chrome.storage) {
      throw new Error("Chrome storage not available");
    }
    await chrome.storage.local.set({
      [COPILOT_AUTH_STORAGE_KEY]: JSON.stringify(auth),
      'vibeApiKey_github-copilot': auth.access || 'oauth-connected'
    });
  },

  async clear(): Promise<void> {
    if (typeof chrome === 'undefined' || !chrome.storage) {
      return;
    }
    await chrome.storage.local.remove([COPILOT_AUTH_STORAGE_KEY, 'vibeApiKey_github-copilot']);
  }
};
