/**
 * Claude Code Provider (Anthropic OAuth)
 * 
 * Implements OAuth PKCE authentication for Anthropic Claude Pro/Max subscriptions.
 * Compatible with OpenCode's opencode-anthropic-auth plugin architecture.
 * 
 * Usage:
 *   const auth = new ClaudeCodeAuth();
 *   const { url, callback } = await auth.authorize('max'); // or 'console' for API key
 *   // Redirect user to url, get code from callback URL
 *   const result = await callback(code);
 *   // Use result.access for API calls, result.refresh for token refresh
 */

const ANTHROPIC_CLIENT_ID = "9d1c250a-e61b-44d9-88ed-5944d1962f5e";

// Anthropic beta headers for Claude Code features
const ANTHROPIC_BETA_HEADERS = [
  "oauth-2025-04-20",
  "claude-code-20250219",
  "interleaved-thinking-2025-05-14",
  "fine-grained-tool-streaming-2025-05-14",
];

export interface ClaudeCodeOAuthState {
  type: 'oauth';
  refresh: string;      // Refresh token (long-lived)
  access: string;       // Access token (short-lived)
  expires: number;      // Expiry timestamp in ms
}

export interface AuthorizeResult {
  url: string;
  instructions: string;
  callback: (code: string) => Promise<AuthResult>;
}

export type AuthResult = 
  | { type: 'success'; refresh: string; access: string; expires: number }
  | { type: 'success'; key: string }  // For API key creation flow
  | { type: 'failed' };

/**
 * Generate PKCE challenge and verifier
 * Based on OpenAuth's generatePKCE implementation
 */
async function generatePKCE(): Promise<{ challenge: string; verifier: string }> {
  // Generate random verifier
  const array = new Uint8Array(32);
  crypto.getRandomValues(array);
  const verifier = btoa(String.fromCharCode(...array))
    .replace(/\+/g, '-')
    .replace(/\//g, '_')
    .replace(/=/g, '');

  // Generate challenge from verifier
  const encoder = new TextEncoder();
  const data = encoder.encode(verifier);
  const hash = await crypto.subtle.digest('SHA-256', data);
  const hashArray = new Uint8Array(hash);
  const challenge = btoa(String.fromCharCode(...hashArray))
    .replace(/\+/g, '-')
    .replace(/\//g, '_')
    .replace(/=/g, '');

  return { challenge, verifier };
}

/**
 * Exchange authorization code for tokens
 */
async function exchangeCode(code: string, verifier: string): Promise<AuthResult> {
  const splits = code.split("#");
  const result = await fetch("https://console.anthropic.com/v1/oauth/token", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      code: splits[0],
      state: splits[1],
      grant_type: "authorization_code",
      client_id: ANTHROPIC_CLIENT_ID,
      redirect_uri: "https://console.anthropic.com/oauth/code/callback",
      code_verifier: verifier,
    }),
  });

  if (!result.ok) {
    return { type: "failed" };
  }

  const json = await result.json();
  return {
    type: "success",
    refresh: json.refresh_token,
    access: json.access_token,
    expires: Date.now() + json.expires_in * 1000,
  };
}

/**
 * Claude Code Authentication Handler
 * Implements OAuth PKCE flow for Anthropic
 */
export class ClaudeCodeAuth {
  /**
   * Authorize with Claude Pro/Max subscription
   * @param mode - 'max' for Claude.ai subscription, 'console' for API key creation
   */
  async authorize(mode: 'max' | 'console' = 'max'): Promise<AuthorizeResult> {
    const pkce = await generatePKCE();

    const baseUrl = mode === "console" 
      ? "https://console.anthropic.com/oauth/authorize"
      : "https://claude.ai/oauth/authorize";

    const url = new URL(baseUrl);
    url.searchParams.set("code", "true");
    url.searchParams.set("client_id", ANTHROPIC_CLIENT_ID);
    url.searchParams.set("response_type", "code");
    url.searchParams.set(
      "redirect_uri",
      "https://console.anthropic.com/oauth/code/callback"
    );
    url.searchParams.set(
      "scope",
      "org:create_api_key user:profile user:inference"
    );
    url.searchParams.set("code_challenge", pkce.challenge);
    url.searchParams.set("code_challenge_method", "S256");
    url.searchParams.set("state", pkce.verifier);

    if (mode === 'console') {
      // API key creation flow
      return {
        url: url.toString(),
        instructions: "Authorize and paste the code here: ",
        callback: async (code: string): Promise<AuthResult> => {
          const credentials = await exchangeCode(code, pkce.verifier);
          if (credentials.type === "failed") return credentials;
          
          // Create API key from OAuth token
          try {
            const response = await fetch(
              "https://api.anthropic.com/api/oauth/claude_cli/create_api_key",
              {
                method: "POST",
                headers: {
                  "Content-Type": "application/json",
                  authorization: `Bearer ${(credentials as any).access}`,
                },
              }
            );

            if (!response.ok) {
              let bodyText: string | undefined;
              try {
                bodyText = await response.text();
              } catch {
                bodyText = undefined;
              }
              const message = `API key creation failed: ${response.status} ${response.statusText}${bodyText ? `: ${bodyText}` : ""}`;
              return { type: "failed", error: message } as AuthResult;
            }

            const result = await response.json();
            return { type: "success", key: (result as any).raw_key };
          } catch (err: any) {
            const message = err?.message || String(err);
            return { type: "failed", error: `API key creation failed: ${message}` } as AuthResult;
          }
        },
      };
    }

    // Claude Pro/Max subscription flow
    return {
      url: url.toString(),
      instructions: "Authorize and paste the code here: ",
      callback: async (code: string): Promise<AuthResult> => {
        return exchangeCode(code, pkce.verifier);
      },
    };
  }

  /**
   * Refresh the access token using the refresh token
   */
  async refreshToken(auth: ClaudeCodeOAuthState): Promise<ClaudeCodeOAuthState> {
    const response = await fetch(
      "https://console.anthropic.com/v1/oauth/token",
      {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          grant_type: "refresh_token",
          refresh_token: auth.refresh,
          client_id: ANTHROPIC_CLIENT_ID,
        }),
      }
    );

    if (!response.ok) {
      throw new Error(`Token refresh failed: ${response.status}`);
    }

    const json = await response.json();
    return {
      type: 'oauth',
      refresh: json.refresh_token,
      access: json.access_token,
      expires: Date.now() + json.expires_in * 1000,
    };
  }
}

/**
 * Create a custom fetch function for Claude Code (Anthropic OAuth)
 * Handles automatic token refresh and proper headers
 */
export function createClaudeCodeFetch(
  getAuth: () => Promise<ClaudeCodeOAuthState | null>,
  saveAuth: (auth: ClaudeCodeOAuthState) => Promise<void>
): typeof fetch {
  const claudeAuth = new ClaudeCodeAuth();

  return async (input: RequestInfo | URL, init?: RequestInit): Promise<Response> => {
    let auth = await getAuth();
    
    if (!auth || auth.type !== 'oauth') {
      throw new Error("Not logged in to Claude. Please connect in Settings.");
    }

    if (!auth.refresh) {
      throw new Error("Claude refresh token missing. Please reconnect in Settings.");
    }

    // Refresh token if expired
    if (!auth.access || auth.expires < Date.now()) {
      auth = await claudeAuth.refreshToken(auth);
      await saveAuth(auth);
    }

    // Merge beta headers
    const existingBeta = (init?.headers as Record<string, string>)?.["anthropic-beta"] || "";
    const existingBetas = existingBeta.split(",").map(b => b.trim()).filter(Boolean);
    const mergedBetas = [...new Set([...ANTHROPIC_BETA_HEADERS, ...existingBetas])].join(",");

    // Build headers
    const headers: Record<string, string> = {
      ...(init?.headers as Record<string, string> || {}),
      authorization: `Bearer ${auth.access}`,
      "anthropic-beta": mergedBetas,
    };

    // Remove conflicting headers
    delete headers["x-api-key"];

    return fetch(input, {
      ...init,
      headers,
    });
  };
}

/**
 * Storage key for Claude Code OAuth
 */
export const CLAUDE_CODE_AUTH_STORAGE_KEY = "vibe_claude_code_oauth";

/**
 * Chrome storage helpers for Claude Code auth
 */
export const ClaudeCodeStorage = {
  async get(): Promise<ClaudeCodeOAuthState | null> {
    if (typeof chrome === 'undefined' || !chrome.storage) {
      return null;
    }
    const result = await chrome.storage.local.get(CLAUDE_CODE_AUTH_STORAGE_KEY);
    const storedValue = result[CLAUDE_CODE_AUTH_STORAGE_KEY];
    if (!storedValue) return null;
    try {
      return JSON.parse(storedValue);
    } catch {
      return null;
    }
  },

  async set(auth: ClaudeCodeOAuthState): Promise<void> {
    if (typeof chrome === 'undefined' || !chrome.storage) {
      throw new Error("Chrome storage not available");
    }
    await chrome.storage.local.set({
      [CLAUDE_CODE_AUTH_STORAGE_KEY]: JSON.stringify(auth),
      'vibeApiKey_anthropic': auth.access || 'oauth-connected'
    });
  },

  async clear(): Promise<void> {
    if (typeof chrome === 'undefined' || !chrome.storage) {
      return;
    }
    await chrome.storage.local.remove([CLAUDE_CODE_AUTH_STORAGE_KEY, 'vibeApiKey_anthropic']);
  }
};
