/**
 * Chrome Nano LLM - LangChain-Compatible Wrapper
 *
 * Makes Chrome's built-in Gemini Nano look like any other LangChain LLM.
 * Uses XML tool emulation since Gemini Nano doesn't support native function calling.
 */

import { AIMessage, BaseMessage } from "@langchain/core/messages";
import { Runnable, RunnableConfig } from "@langchain/core/runnables";
import { XmlToolParser, extractParamNames, ToolCall } from './XmlToolParser.js';

// Declare LanguageModel global for Chrome 145+
declare const LanguageModel: {
  availability: () => Promise<string>;
  create: (options: AILanguageModelCreateOptions) => Promise<AILanguageModelSession>;
} | undefined;

const XML_TOOL_PROMPT = `====

TOOL USE

You have access to tools that help you complete tasks. To use a tool, format your response with XML tags:

<tool_name>
<parameter1>value1</parameter1>
<parameter2>value2</parameter2>
</tool_name>

IMPORTANT RULES:
- Use exactly ONE tool per response
- After the tool executes, you'll see the result and can continue
- Always include all required parameters
- The tool name must match exactly

Available tools:
{tool_descriptions}

====`;

interface LangChainTool {
  name: string;
  description?: string;
  schema?: {
    shape?: Record<string, unknown>;
    _def?: {
      shape: () => Record<string, unknown>;
    };
  };
  parameters?: {
    properties?: Record<string, unknown>;
  };
}

interface ChromeNanoConfig {
  temperature?: number;
  topK?: number;
}

interface MessageInput {
  messages?: BaseMessage[];
}

interface StructuredOutputOptions {
  name?: string;
}

/**
 * LangChain-compatible wrapper for Chrome's Gemini Nano
 */
export class ChromeNanoLLM extends Runnable<MessageInput | BaseMessage[], AIMessage> {
  lc_namespace = ["langchain", "llms", "nano"];
  
  private temperature: number;
  private topK: number;
  private session: AILanguageModelSession | null = null;
  private tools: LangChainTool[] = [];
  private toolParser: XmlToolParser | null = null;

  constructor(config: ChromeNanoConfig = {}) {
    super(config);
    this.temperature = config.temperature ?? 0.7;
    this.topK = config.topK ?? 40;
  }

  // LangChain serialization (inherited from Runnable)

  /**
   * Bind tools to this LLM (LangChain interface)
   */
  bindTools(tools: LangChainTool[]): this {
    this.tools = tools || [];

    if (this.tools.length > 0) {
      const toolNames = this.tools.map(t => t.name);
      const paramNames = extractParamNames(this.tools);
      this.toolParser = new XmlToolParser(toolNames, paramNames);
    } else {
      this.toolParser = null;
    }

    return this;
  }

  /**
   * Create a Nano session using available API
   */
  private async createSession(): Promise<AILanguageModelSession> {
    const options: AILanguageModelCreateOptions = {
      temperature: this.temperature,
      topK: this.topK
    };

    // NEW API (Chrome 145+): LanguageModel global
    if (typeof LanguageModel !== 'undefined') {
      return LanguageModel.create(options);
    }

    // OLD API: self.ai.languageModel
    if (typeof self !== 'undefined' && self.ai?.languageModel) {
      return self.ai.languageModel.create(options);
    }

    throw new Error('Chrome Built-in AI not available. Enable the #prompt-api-for-gemini-nano flag in chrome://flags.');
  }

  /**
   * Invoke the LLM (LangChain interface)
   */
  async invoke(input: MessageInput | BaseMessage[], _options?: RunnableConfig): Promise<AIMessage> {
    if (!this.session) {
      this.session = await this.createSession();
    }

    let messages: BaseMessage[];
    if (Array.isArray(input)) {
      messages = input;
    } else if ('messages' in input && Array.isArray(input.messages)) {
      messages = input.messages;
    } else {
      throw new Error(
        'Invalid input to ChromeNanoLLM.invoke: expected BaseMessage[] or MessageInput with a messages array'
      );
    }
    let prompt = this.messagesToPrompt(messages);

    if (this.tools.length > 0) {
      const toolDescriptions = this.tools.map(t => {
        const params = this.getToolParams(t);
        return `- ${t.name}: ${t.description || 'No description'}\n  Parameters: ${params}`;
      }).join('\n');

      prompt = XML_TOOL_PROMPT.replace('{tool_descriptions}', toolDescriptions) + '\n\n' + prompt;
    }

    let response: string;
    try {
      response = await this.session.prompt(prompt);
    } catch (error) {
      const err = error as Error & { name?: string };
      // Check for session invalid/destroyed errors
      // Chrome's LanguageModel API surfaces these as DOMException-like errors
      const isSessionInvalidError =
        (typeof DOMException !== 'undefined' &&
          err instanceof DOMException &&
          (err.name === 'InvalidStateError' || err.name === 'NotFoundError')) ||
        err.name === 'InvalidStateError' ||
        err.message?.includes('destroyed') ||
        err.message?.includes('invalid');

      if (isSessionInvalidError) {
        this.session = await this.createSession();
        response = await this.session.prompt(prompt);
      } else {
        throw error;
      }
    }

    let toolCalls: ToolCall[] = [];
    if (this.toolParser) {
      this.toolParser.reset();
      this.toolParser.processChunk(response);
      toolCalls = this.toolParser.getToolCalls();
    }

    return new AIMessage({
      content: response,
      tool_calls: toolCalls.length > 0 ? toolCalls : undefined,
      additional_kwargs: { app_type: 'thought' },
      response_metadata: {
        finish_reason: toolCalls.length > 0 ? 'tool_calls' : 'stop',
        model: 'gemini-nano'
      }
    });
  }

  /**
   * Convert LangChain messages to prompt string
   */
  private messagesToPrompt(messages: BaseMessage[] | unknown): string {
    if (!Array.isArray(messages)) {
      if (typeof messages === 'string') return messages;
      if (messages && typeof messages === 'object' && 'content' in messages) {
        return String((messages as { content: unknown }).content);
      }
      return JSON.stringify(messages);
    }

    return messages.map(msg => {
      const type = (msg as { _getType?: () => string })._getType?.() || 
                   (msg as { type?: string }).type || 'unknown';
      let content = '';

      if (typeof msg.content === 'string') {
        content = msg.content;
      } else if (Array.isArray(msg.content)) {
        interface ContentPart { type: string; text?: string }
        content = (msg.content as ContentPart[])
          .filter((part: ContentPart) => part.type === 'text' && part.text)
          .map((part: ContentPart) => part.text!)
          .join('\n');
      } else if (msg.content) {
        content = JSON.stringify(msg.content);
      }

      const toolCalls = (msg as { tool_calls?: Array<{ name: string }> }).tool_calls;
      const toolName = (msg as { name?: string }).name;

      switch (type) {
        case 'system':
          return `System: ${content}`;
        case 'human':
          return `User: ${content}`;
        case 'ai':
          if (toolCalls?.length) {
            const toolCallStr = toolCalls.map(tc => `[Called ${tc.name}]`).join(' ');
            return `Assistant: ${content} ${toolCallStr}`;
          }
          return `Assistant: ${content}`;
        case 'tool':
          return `Tool Result (${toolName}): ${content}`;
        default:
          return content;
      }
    }).join('\n\n');
  }

  private getToolParams(tool: LangChainTool): string {
    if (tool.schema?.shape) {
      return JSON.stringify(Object.keys(tool.schema.shape));
    }
    if (tool.schema?._def?.shape) {
      return JSON.stringify(Object.keys(tool.schema._def.shape()));
    }
    if (tool.parameters?.properties) {
      return JSON.stringify(Object.keys(tool.parameters.properties));
    }
    return '{}';
  }

  /**
   * Structured output wrapper (for reflection)
   */
  withStructuredOutput(_schema: unknown, options: StructuredOutputOptions = {}): { invoke: (input: BaseMessage[]) => Promise<unknown> } {
    return {
      invoke: async (input: BaseMessage[]) => {
        const schemaHint = options.name
          ? `Respond with valid JSON matching the "${options.name}" schema.`
          : 'Respond with valid JSON.';

        const messages = Array.isArray(input) ? input : [input];
        const enhancedInput = [...messages, { content: schemaHint, _getType: () => 'system' }] as BaseMessage[];

        const response = await this.invoke(enhancedInput);

        try {
          return JSON.parse(response.content as string);
        } catch (parseError) {
          // Log initial JSON parsing error for debugging
          console.warn('ChromeNanoLLM.withStructuredOutput: initial JSON.parse failed', {
            error: parseError instanceof Error ? parseError.message : String(parseError),
            contentPreview: (response.content as string).slice(0, 200)
          });

          const jsonMatch = (response.content as string).match(/\{[\s\S]*\}/);
          if (jsonMatch) {
            return JSON.parse(jsonMatch[0]);
          }
          throw new Error(`Failed to parse structured output: ${(response.content as string).slice(0, 100)}`);
        }
      }
    };
  }

  destroy(): void {
    if (this.session) {
      try {
        this.session.destroy();
      } catch (error) {
        // Session might already be destroyed, log for debugging
        console.warn('ChromeNanoLLM: Failed to destroy session:', error);
      }
      this.session = null;
    }
  }
}

export default ChromeNanoLLM;
