// Popup helper for showing reasoning during tool execution
export class PopupHelper {
  static _trustedTypesSupported = null;

  // Check if the page supports Trusted Types
  static _checkTrustedTypesSupport() {
    if (this._trustedTypesSupported === null) {
      try {
        // Check if we're in a browser environment with Trusted Types API
        if (typeof window !== 'undefined' && window.trustedTypes && window.trustedTypes.createPolicy) {
          // Try to create a test policy to see if we can use it
          const testPolicy = window.trustedTypes.createPolicy('vibe-test-policy', {
            createHTML: () => 'test'
          });
          if (testPolicy && typeof testPolicy.createHTML === 'function') {
            this._trustedTypesSupported = true;
          } else {
            this._trustedTypesSupported = false;
          }
        } else {
          this._trustedTypesSupported = false;
        }
      } catch (error) {
        this._trustedTypesSupported = false;
      }
    }
    return this._trustedTypesSupported;
  }

  static async showReasoningPopup(page, reasoning, toolName) {
    try {
      // Skip popup injection if we've previously failed on this page
      // This prevents retry loops that cause rate limiting
      if (this._popupDisabled) {
        console.debug('Popup injection disabled due to previous TrustedHTML failure');
        return;
      }

      // Inject popup into the page with Trusted Types support
      await page.evaluate(({ reasoning, toolName }) => {
        try {
          // Helper function to safely set HTML content
          const setSafeHTML = (element, html) => {
            // Check for Trusted Types support
            if (typeof window !== 'undefined' &&
                window.trustedTypes &&
                window.trustedTypes.createPolicy &&
                element.isConnected) {

              try {
                // Create or get existing Trusted Types policy
                let policy = window.trustedTypes.getPolicy('vibe-reasoning-policy');
                if (!policy) {
                  policy = window.trustedTypes.createPolicy('vibe-reasoning-policy', {
                    createHTML: (html) => html
                  });
                }
                // Use trusted types to set HTML
                element.innerHTML = policy.createHTML(html);
                return true;
              } catch (trustedError) {
                console.warn('Trusted Types policy failed:', trustedError.message);
                return false;
              }
            } else {
              // Fall back to direct setting if Trusted Types not available
              try {
                element.innerHTML = html;
                return true;
              } catch (directError) {
                console.warn('Direct HTML setting failed:', directError.message);
                return false;
              }
            }
          };

          // Remove any existing popup
          const existingPopup = document.getElementById('vibe-reasoning-popup');
          if (existingPopup) {
            existingPopup.remove();
          }

          // Create popup container using safe method
          const popup = document.createElement('div');
          popup.id = 'vibe-reasoning-popup';

          const popupHTML = `
            <div style="
              position: fixed;
              top: 20px;
              right: 20px;
              max-width: 400px;
              background: rgba(255, 255, 255, 0.95);
              border: 1px solid #e0e0e0;
              border-radius: 8px;
              padding: 16px;
              box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
              z-index: 999999;
              font-family: 'Roboto', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
              animation: slideIn 0.3s ease-out;
              opacity: 0.75;
            ">
              <style>
                @keyframes slideIn {
                  from {
                    transform: translateX(100%);
                    opacity: 0;
                  }
                  to {
                    transform: translateX(0);
                    opacity: 0.75;
                  }
                }
                @keyframes fadeOut {
                  from {
                    opacity: 0.75;
                  }
                  to {
                    opacity: 0;
                  }
                }
              </style>

              <button
                onclick="(() => {
                  const popup = document.getElementById('vibe-reasoning-popup');
                  if (popup) {
                    popup.style.animation = 'fadeOut 0.3s ease-out';
                    setTimeout(() => popup.remove(), 300);
                  }
                })()"
                style="
                  position: absolute;
                  top: 8px;
                  right: 8px;
                  background: none;
                  border: none;
                  font-size: 18px;
                  color: #999;
                  cursor: pointer;
                  padding: 4px;
                  line-height: 1;
                ">×</button>

              <div style="
                font-size: 12px;
                font-weight: 500;
                color: #666;
                margin-bottom: 8px;
                text-transform: uppercase;
                letter-spacing: 0.5px;
              ">AI Agent - ${toolName}</div>

              <div style="
                font-size: 14px;
                line-height: 1.5;
                color: #333;
              ">${reasoning}</div>
            </div>
          `;

          // Try to set the HTML content safely
          if (!setSafeHTML(popup, popupHTML)) {
            throw new Error('Failed to set popup HTML content - Trusted Types or security policy issue');
          }

          document.body.appendChild(popup);

          // Auto-hide after 5 seconds
          setTimeout(() => {
            if (document.getElementById('vibe-reasoning-popup')) {
              popup.style.animation = 'fadeOut 0.3s ease-out';
              setTimeout(() => popup.remove(), 300);
            }
          }, 5000);
        } catch (error) {
          // Re-throw to be caught by outer try-catch
          throw error;
        }
      }, { reasoning, toolName });
    } catch (error) {
      // Mark popup as disabled to prevent retry loops
      this._popupDisabled = true;
      console.debug('Could not show reasoning popup (non-retryable):', error.message);
    }
  }

  static async hidePopup(page) {
    try {
      await page.evaluate(() => {
        const popup = document.getElementById('vibe-reasoning-popup');
        if (popup) {
          popup.style.animation = 'fadeOut 0.3s ease-out';
          setTimeout(() => popup.remove(), 300);
        }
      });
    } catch (error) {
      // Silently fail
      console.debug('Could not hide popup:', error.message);
    }
  }

  // Reset popup disabled state for new pages/sessions
  static resetPopupState() {
    this._popupDisabled = false;
    this._trustedTypesSupported = null; // Reset cached trust types check
  }
}
