import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import fs from "fs/promises";
import path from "path";

/**
 * Upload File Tool - Handle file uploads to web forms
 * Inspired by browser-use's upload_file action
 */
export class PlaywrightUploadFileTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "upload_file",
      "Upload a file to a file input element on the page",
      z.object({
        filePath: z.string().describe("Path to the file to upload"),
        selector: z.string()
          .optional()
          .describe("CSS selector for the file input (optional - will try to find automatically)")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ filePath, selector }) {
    try {
      // Resolve file path
      const resolvedPath = path.resolve(filePath);
      
      if (this.options.verbose) {
        console.log(`📁 Preparing to upload file: ${resolvedPath}`);
      }

      // Check if file exists
      try {
        const stats = await fs.stat(resolvedPath);
        if (!stats.isFile()) {
          throw new Error(`Path is not a file: ${resolvedPath}`);
        }
        
        if (this.options.verbose) {
          console.log(`   File size: ${(stats.size / 1024).toFixed(2)} KB`);
        }
      } catch (error) {
        throw new Error(`File not found or inaccessible: ${resolvedPath}`);
      }

      // Find file input element
      let fileInput;
      
      if (selector) {
        // Use provided selector
        fileInput = this.page.locator(selector).first();
        
        // Verify it's a file input
        const tagName = await fileInput.evaluate(el => el.tagName);
        const type = await fileInput.evaluate(el => el.type).catch(() => null);
        
        if (tagName.toLowerCase() !== 'input' || type !== 'file') {
          console.warn(`⚠️ Selected element may not be a file input: ${tagName}[type=${type}]`);
        }
      } else {
        // Try to find file input automatically
        fileInput = this.page.locator('input[type="file"]').first();
        
        // Check if found
        const count = await this.page.locator('input[type="file"]').count();
        if (count === 0) {
          throw new Error("No file input found on the page. Please provide a selector.");
        }
        
        if (count > 1 && this.options.verbose) {
          console.log(`   Found ${count} file inputs, using the first one`);
        }
      }

      // Set the file
      await fileInput.setInputFiles(resolvedPath);
      
      // Wait a moment for any upload handlers
      await this.page.waitForTimeout(500);

      // Try to detect if upload triggered any changes
      let uploadStatus = "File selected";
      
      // Check for common upload feedback elements
      const feedbackSelectors = [
        '.upload-success',
        '.file-uploaded',
        '[class*="success"]',
        '[class*="uploaded"]',
        '.filename',
        '.file-name'
      ];
      
      for (const feedbackSelector of feedbackSelectors) {
        const feedback = await this.page.locator(feedbackSelector).first();
        if (await feedback.isVisible().catch(() => false)) {
          const text = await feedback.textContent().catch(() => "");
          if (text && text.includes(path.basename(resolvedPath))) {
            uploadStatus = `File uploaded: ${text}`;
            break;
          }
        }
      }

      const message = `${uploadStatus} - ${path.basename(resolvedPath)}`;
      
      if (this.options.verbose) {
        console.log(`${message}`);
      }

      return message;

    } catch (error) {
      const errorMsg = `Failed to upload file: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * Create Test File Tool - Create a temporary file for testing uploads
 */
export class PlaywrightCreateTestFileTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "create_test_file",
      "Create a temporary test file for upload testing",
      z.object({
        fileName: z.string()
          .optional()
          .default("test-file.txt")
          .describe("Name for the test file"),
        content: z.string()
          .optional()
          .default("This is a test file created for upload testing.")
          .describe("Content to write to the file"),
        directory: z.string()
          .optional()
          .default("./temp")
          .describe("Directory to create the file in")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ fileName = "test-file.txt", content = "This is a test file.", directory = "./temp" }) {
    try {
      // Ensure directory exists
      const dirPath = path.resolve(directory);
      await fs.mkdir(dirPath, { recursive: true });
      
      // Create file path
      const filePath = path.join(dirPath, fileName);
      
      // Write content to file
      await fs.writeFile(filePath, content, 'utf-8');
      
      if (this.options.verbose) {
        console.log(`📝 Created test file: ${filePath}`);
      }

      return `Created test file: ${filePath}`;

    } catch (error) {
      const errorMsg = `Failed to create test file: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * Handle Download Tool - Monitor and handle file downloads
 */
export class PlaywrightHandleDownloadTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "handle_download",
      "Click a download link and save the downloaded file",
      z.object({
        selector: z.string().describe("Selector for the download link/button"),
        saveAs: z.string()
          .optional()
          .describe("Path to save the downloaded file (optional)"),
        timeout: z.number()
          .optional()
          .default(30000)
          .describe("Maximum time to wait for download in milliseconds")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ selector, saveAs, timeout = 30000 }) {
    try {
      // Start waiting for download before clicking
      const downloadPromise = this.page.waitForEvent('download', { timeout });
      
      // Click the download trigger
      await this.page.locator(selector).first().click();
      
      if (this.options.verbose) {
        console.log(`⏬ Waiting for download...`);
      }
      
      // Wait for download to start
      const download = await downloadPromise;
      
      // Get suggested filename
      const suggestedFilename = download.suggestedFilename();
      
      // Determine save path
      const savePath = saveAs 
        ? path.resolve(saveAs)
        : path.resolve('./downloads', suggestedFilename);
      
      // Ensure directory exists
      await fs.mkdir(path.dirname(savePath), { recursive: true });
      
      // Save the download
      await download.saveAs(savePath);
      
      // Get file size
      const stats = await fs.stat(savePath);
      const sizeKB = (stats.size / 1024).toFixed(2);
      
      const message = `Downloaded ${suggestedFilename} (${sizeKB} KB) to ${savePath}`;
      
      if (this.options.verbose) {
        console.log(`${message}`);
      }

      return message;

    } catch (error) {
      const errorMsg = `Download failed: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}