import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

/**
 * Universal Search Tool - Multi-engine search with fallback strategy
 * Inspired by browser-use MCP implementations
 * 
 * Strategy:
 * 1. Try DuckDuckGo (most reliable, no bot detection)
 * 2. Fallback to Bing (less strict than Google)
 * 3. Last resort: Google (often blocks bots)
 */
export class PlaywrightUniversalSearchTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "search_web",
      "Search the web using multiple search engines with automatic fallback. Tries DuckDuckGo first, then Bing, then Google.",
      z.object({
        query: z.string().describe("The search query"),
        engine: z.enum(['auto', 'duckduckgo', 'bing', 'google']).optional().default('auto')
          .describe("Search engine to use (auto will try multiple engines)")
      })
    );
    this.page = page;
    this.options = options;
    
    // Search engine configurations
    this.engines = {
      duckduckgo: {
        name: 'DuckDuckGo',
        url: (q) => `https://duckduckgo.com/?q=${encodeURIComponent(q)}`,
        resultSelectors: [
          '[data-testid="result"]',
          '.result',
          '[data-nrn="result"]',
          'article[data-testid="result"]',
          '.react-results--main article',
          '.results--main article'
        ],
        titleSelector: 'h2 a, .result__title a',
        snippetSelector: '.result__snippet, [data-result="snippet"]',
        linkSelector: 'h2 a, .result__title a'
      },
      bing: {
        name: 'Bing',
        url: (q) => `https://www.bing.com/search?q=${encodeURIComponent(q)}`,
        resultSelectors: [
          '.b_algo',
          '.b_result',
          'li.b_algo'
        ],
        titleSelector: 'h2 a, .b_title a',
        snippetSelector: '.b_caption p, .b_snippet',
        linkSelector: 'h2 a'
      },
      google: {
        name: 'Google',
        url: (q) => `https://www.google.com/search?q=${encodeURIComponent(q)}&udm=14`,
        resultSelectors: [
          'div.g',
          '[data-sokoban-container]',
          '.tF2Cxc'
        ],
        titleSelector: 'h3',
        snippetSelector: '.VwiC3b, .yXK7lf',
        linkSelector: 'a[href]'
      }
    };
  }

  async searchWithEngine(engineName, query) {
    const engine = this.engines[engineName];
    const searchUrl = engine.url(query);
    
    if (this.options.verbose) {
      console.log(`🔍 Trying ${engine.name} search for: "${query}"`);
    }
    
    try {
      // Navigate to search URL
      await this.page.goto(searchUrl, { 
        waitUntil: 'domcontentloaded',
        timeout: 15000 
      });
      
      // Wait a bit for results to load
      await this.page.waitForTimeout(1500);
      
      // Check if we hit a bot detection page
      const url = this.page.url();
      if (url.includes('/sorry') || url.includes('captcha') || url.includes('blocked') || 
          url.includes('/static-pages/418') || url.includes('error')) {
        throw new Error(`Bot detection triggered on ${engine.name}`);
      }
      
      // Try to find search results
      let resultsFound = false;
      for (const selector of engine.resultSelectors) {
        const count = await this.page.locator(selector).count();
        if (count > 0) {
          resultsFound = true;
          if (this.options.verbose) {
            console.log(`   Found ${count} results using selector: ${selector}`);
          }
          break;
        }
      }
      
      if (!resultsFound) {
        throw new Error(`No results found on ${engine.name}`);
      }
      
      // Extract first few results for context
      const results = await this.extractResults(engine, 3);
      
      return {
        success: true,
        engine: engine.name,
        url: this.page.url(),
        message: `Successfully searched ${engine.name} for "${query}"`,
        results: results
      };
      
    } catch (error) {
      if (this.options.verbose) {
        console.log(`   ❌ ${engine.name} failed: ${error.message}`);
      }
      throw error;
    }
  }
  
  async extractResults(engine, limit = 3) {
    const results = [];
    
    try {
      // Try each result selector until we find one that works
      for (const selector of engine.resultSelectors) {
        const elements = await this.page.locator(selector).all();
        
        if (elements.length > 0) {
          for (let i = 0; i < Math.min(elements.length, limit); i++) {
            const element = elements[i];
            
            try {
              const title = await element.locator(engine.titleSelector).first().textContent() || '';
              const snippet = await element.locator(engine.snippetSelector).first().textContent() || '';
              const link = await element.locator(engine.linkSelector).first().getAttribute('href') || '';
              
              if (title) {
                results.push({
                  title: title.trim(),
                  snippet: snippet.trim(),
                  link: link
                });
              }
            } catch (e) {
              // Skip this result if extraction fails
              continue;
            }
          }
          
          if (results.length > 0) {
            break; // Found results, stop trying other selectors
          }
        }
      }
    } catch (error) {
      console.warn('Failed to extract detailed results:', error.message);
    }
    
    return results;
  }

  async call({ query, engine = 'auto' }) {
    if (engine !== 'auto') {
      // Use specific engine
      try {
        const result = await this.searchWithEngine(engine, query);
        return this.formatResult(result);
      } catch (error) {
        throw new Error(`Search failed with ${engine}: ${error.message}`);
      }
    }
    
    // Auto mode - try engines in order
    const engineOrder = ['duckduckgo', 'bing', 'google'];
    const errors = [];
    
    for (const engineName of engineOrder) {
      try {
        const result = await this.searchWithEngine(engineName, query);
        if (this.options.verbose) {
          console.log(`Search succeeded with ${engineName}`);
        }
        return this.formatResult(result);
      } catch (error) {
        errors.push(`${engineName}: ${error.message}`);
        continue; // Try next engine
      }
    }
    
    // All engines failed
    const errorMsg = `All search engines failed:\n${errors.join('\n')}`;
    console.error(`❌ ${errorMsg}`);
    throw new Error(errorMsg);
  }
  
  formatResult(result) {
    let message = result.message;
    
    if (result.results && result.results.length > 0) {
      message += '\n\nTop results:';
      result.results.forEach((r, i) => {
        message += `\n${i + 1}. ${r.title}`;
        if (r.snippet) {
          message += `\n   ${r.snippet.substring(0, 100)}...`;
        }
      });
    }
    
    return message;
  }
}

// Export a simplified version that uses the universal search by default
export class PlaywrightSearchTool extends PlaywrightUniversalSearchTool {
  constructor(page, options = {}) {
    super(page, options);
    this.name = "search"; // Simplified name
    this.description = "Search the web with automatic fallback across multiple search engines";
  }
}