import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

/**
 * Tavily Search Tool - AI-optimized search engine
 * Best search tool for AI agents - returns clean, structured results
 * 
 * Get your FREE API key at: https://tavily.com
 * - 1000 free searches per month
 * - No credit card required
 * - Optimized for LLMs
 */
export class PlaywrightTavilySearchTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "search_tavily",
      "Search the web using Tavily AI-optimized search engine. Returns high-quality, structured results perfect for AI agents.",
      z.object({
        query: z.string().describe("The search query"),
        searchDepth: z.enum(['basic', 'advanced']).optional().default('basic')
          .describe("Search depth - 'basic' for quick results, 'advanced' for comprehensive"),
        maxResults: z.number().optional().default(5).describe("Maximum number of results"),
        includeAnswer: z.boolean().optional().default(true)
          .describe("Include AI-generated answer to the query"),
        includeRawContent: z.boolean().optional().default(false)
          .describe("Include raw page content (uses more tokens)")
      })
    );
    this.page = page; // Keep for compatibility
    this.options = options;
    this.apiKey = options.apiKey || process.env.TAVILY_API_KEY;
  }

  async call({ query, searchDepth = 'basic', maxResults = 5, includeAnswer = true, includeRawContent = false }) {
    if (!this.apiKey) {
      throw new Error(
        "Tavily API key not found!\n" +
        "Please set TAVILY_API_KEY environment variable or pass apiKey in options.\n" +
        "Get your FREE API key at: https://tavily.com (1000 searches/month free)"
      );
    }

    try {
      if (this.options.verbose) {
        console.log(`🔍 Searching with Tavily: "${query}"`);
      }

      // Tavily API endpoint
      const response = await fetch('https://api.tavily.com/search', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'api-key': this.apiKey
        },
        body: JSON.stringify({
          query: query,
          search_depth: searchDepth,
          max_results: maxResults,
          include_answer: includeAnswer,
          include_raw_content: includeRawContent,
          include_images: false,
          api_key: this.apiKey
        })
      });

      if (!response.ok) {
        const error = await response.text();
        throw new Error(`Tavily API error: ${response.status} - ${error}`);
      }

      const data = await response.json();

      if (this.options.verbose) {
        console.log(`Found ${data.results?.length || 0} results`);
      }

      // Format the response
      return this.formatResults(data, query);

    } catch (error) {
      const errorMsg = `Tavily search failed: ${error.message}`;
      if (this.options.verbose) {
        console.error(`❌ ${errorMsg}`);
      }
      throw new Error(errorMsg);
    }
  }

  formatResults(data, query) {
    let output = `Tavily Search Results for "${query}"\n`;
    output += `${'='.repeat(50)}\n\n`;

    // Include the AI-generated answer if available
    if (data.answer) {
      output += `📝 AI Answer:\n${data.answer}\n\n`;
      output += `${'─'.repeat(40)}\n\n`;
    }

    // Include search results
    if (data.results && data.results.length > 0) {
      output += `📊 Search Results (${data.results.length}):\n\n`;
      
      data.results.forEach((result, index) => {
        output += `${index + 1}. ${result.title}\n`;
        output += `   🔗 ${result.url}\n`;
        
        if (result.score) {
          output += `   📈 Relevance: ${(result.score * 100).toFixed(1)}%\n`;
        }
        
        if (result.snippet || result.content) {
          const text = result.snippet || result.content;
          output += `   📄 ${text.substring(0, 200)}...\n`;
        }
        
        if (result.published_date) {
          output += `   📅 Published: ${result.published_date}\n`;
        }
        
        output += '\n';
      });
    }

    // Include query metadata
    if (data.query) {
      output += `${'─'.repeat(40)}\n`;
      output += `🔎 Query: "${data.query}"\n`;
      output += `⏱️ Search depth: ${data.search_depth || 'basic'}\n`;
      output += `📍 Results: ${data.results?.length || 0} found\n`;
    }

    return output;
  }
}

/**
 * Simplified Tavily tool with minimal configuration
 */
export class TavilySearchTool extends PlaywrightTavilySearchTool {
  constructor(page, options = {}) {
    super(page, options);
    this.name = "search_web";
    this.description = "Search the web using Tavily (AI-optimized search engine)";
  }

  async call({ query, maxResults = 3 }) {
    // Simplified call with just the essentials
    return super.call({
      query,
      maxResults,
      searchDepth: 'basic',
      includeAnswer: true,
      includeRawContent: false
    });
  }
}

// Export a function to check if Tavily is configured
export function isTavilyConfigured() {
  return !!process.env.TAVILY_API_KEY;
}

// Export a helper to set up Tavily
export function setupTavily(apiKey) {
  if (!apiKey) {
    console.log("🔑 To use Tavily Search:");
    console.log("1. Go to https://tavily.com");
    console.log("2. Sign up for free (takes 30 seconds)");
    console.log("3. Copy your API key");
    console.log("4. Set environment variable: export TAVILY_API_KEY='your-key-here'");
    return false;
  }
  process.env.TAVILY_API_KEY = apiKey;
  return true;
}