import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

export class PlaywrightListTabsTool extends BrowserTool {
  constructor(page) {
    super(
      "list_tabs",
      "List all browser pages/tabs",
      z.object({
        reasoning: z.string().describe("Reasoning for this action")
      })
    );
    this.page = page;
  }

  async call({ reasoning }) {
    try {
      const context = this.page.context();
      const pages = context.pages();
      const pageInfo = await Promise.all(pages.map(async (p, index) => {
        const url = p.url();
        const title = await p.title();
        const isCurrent = p === this.page ? ' [CURRENT]' : '';
        return `Page ${index}${isCurrent}: "${title}" - ${url}`;
      }));
      return `Found ${pages.length} page(s):\n${pageInfo.join('\n')}`;
    } catch (error) {
      throw new Error(`Failed to list pages: ${error.message}`);
    }
  }
}

export class PlaywrightCreateNewTabTool extends BrowserTool {
  constructor(page) {
    super(
      "create_new_tab",
      "Create a new browser tab/page",
      z.object({
        url: z.string().optional().describe("Optional URL to navigate to (with or without protocol)"),
        reasoning: z.string().describe("Reasoning for this action")
      })
    );
    this.page = page;
  }

  async call({ url, reasoning }) {
    try {
      const context = this.page.context();
      const newPage = await context.newPage();
      if (url) {
        // Add protocol if missing
        let fullUrl = url;
        if (!url.match(/^https?:\/\//)) {
          fullUrl = `https://${url}`;
        }
        await newPage.goto(fullUrl, { waitUntil: 'networkidle' });
      }
      // Update the page reference for subsequent tools
      this.page = newPage;
      return `Created new page${url ? ` and navigated to ${url}` : ''}`;
    } catch (error) {
      throw new Error(`Failed to create new tab: ${error.message}`);
    }
  }
}