import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

/**
 * Smart Wait Tool - Intelligent waiting with auto-reduction
 * Inspired by browser-use's wait action that reduces wait time by 3s (LLM processing time)
 */
export class PlaywrightSmartWaitTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "smart_wait",
      "Wait for specified seconds (max 10). Automatically reduces wait time by 3s to account for LLM processing time. Use this to wait for page elements to load or animations to complete.",
      z.object({
        seconds: z.number()
          .min(0)
          .max(10)
          .default(3)
          .describe("Number of seconds to wait (will be auto-reduced by 3s for LLM time)")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ seconds = 3 }) {
    try {
      // Cap wait time at maximum 10 seconds
      const requestedSeconds = Math.min(seconds, 10);
      
      // Reduce wait time by 3 seconds to account for LLM processing time
      // This is because the LLM call itself takes about 3 seconds
      const actualSeconds = Math.max(requestedSeconds - 3, 0);
      
      if (this.options.verbose) {
        console.log(`⏳ Smart wait: requested ${requestedSeconds}s, actual ${actualSeconds}s (reduced by 3s for LLM time)`);
      }

      // Perform the actual wait
      if (actualSeconds > 0) {
        await this.page.waitForTimeout(actualSeconds * 1000);
      }

      const message = actualSeconds === 0 
        ? `Skipped wait (requested ${requestedSeconds}s was less than LLM processing time)`
        : `Waited for ${requestedSeconds} seconds (actual: ${actualSeconds}s)`;

      if (this.options.verbose) {
        console.log(`${message}`);
      }

      return message;

    } catch (error) {
      const errorMsg = `Smart wait failed: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * Traditional Wait Tool - Wait without auto-reduction
 * For cases where exact timing is needed
 */
export class PlaywrightExactWaitTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "exact_wait",
      "Wait for exact specified seconds without auto-reduction. Use when precise timing is required.",
      z.object({
        seconds: z.number()
          .min(0)
          .max(30)
          .default(1)
          .describe("Exact number of seconds to wait")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ seconds = 1 }) {
    try {
      const waitSeconds = Math.min(Math.max(seconds, 0), 30);
      
      if (this.options.verbose) {
        console.log(`⏱️ Exact wait: ${waitSeconds} seconds`);
      }

      await this.page.waitForTimeout(waitSeconds * 1000);

      const message = `Waited exactly ${waitSeconds} seconds`;

      if (this.options.verbose) {
        console.log(`${message}`);
      }

      return message;

    } catch (error) {
      const errorMsg = `Exact wait failed: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * Wait For Load State Tool - Wait for page to reach specific state
 */
export class PlaywrightWaitForLoadStateTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "wait_for_load_state",
      "Wait for page to reach a specific load state (load, domcontentloaded, networkidle).",
      z.object({
        state: z.enum(["load", "domcontentloaded", "networkidle"])
          .default("domcontentloaded")
          .describe("The load state to wait for"),
        timeout: z.number()
          .min(0)
          .max(60000)
          .default(30000)
          .describe("Maximum time to wait in milliseconds")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ state = "domcontentloaded", timeout = 30000 }) {
    try {
      if (this.options.verbose) {
        console.log(`⏳ Waiting for page to reach '${state}' state (timeout: ${timeout}ms)`);
      }

      const startTime = Date.now();
      
      await this.page.waitForLoadState(state, { timeout });
      
      const elapsedTime = Date.now() - startTime;
      const message = `Page reached '${state}' state in ${(elapsedTime / 1000).toFixed(2)} seconds`;

      if (this.options.verbose) {
        console.log(`${message}`);
      }

      return message;

    } catch (error) {
      if (error.message.includes("Timeout")) {
        const errorMsg = `Timeout waiting for '${state}' state after ${timeout}ms`;
        console.error(`❌ ${errorMsg}`);
        throw new Error(errorMsg);
      }
      
      const errorMsg = `Failed to wait for load state: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}