// @ts-nocheck

// SecretsManager tool for reading credentials from .secrets.csv file

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import fs from 'fs/promises';
import path from 'path';
// Simple CSV parser since we don't have csv-parse installed

/**
 * SecretsManager tool for managing credentials stored in .secrets.csv
 * CSV format: service,login,password
 */
export class PlaywrightSecretsManager extends BrowserTool {
  constructor(page = null) {
    super();
    this.page = page;
    this.secretsFile = '.secrets.csv';
    this.secrets = null;
  }

  get name() {
    return "secrets_manager";
  }
  
  set name(value) {
    // No-op setter to avoid errors
  }

  get description() {
    return "Manage credentials and secrets. Operations: 'list' returns all service names, 'read <service>' returns login and password for that service";
  }
  
  set description(value) {
    // No-op setter to avoid errors
  }

  get schema() {
    return z.object({
      operation: z.enum(['list', 'read']).describe("Operation to perform: 'list' or 'read'"),
      service: z.string().optional().describe("Service name (required for 'read' operation)"),
      reasoning: z.string().describe("Why you need to access these credentials")
    });
  }
  
  set schema(value) {
    // No-op setter to avoid errors
  }

  async _loadSecrets() {
    if (this.secrets !== null) {
      return this.secrets;
    }

    try {
      // Look for .secrets.csv in current working directory
      const filePath = path.join(process.cwd(), this.secretsFile);
      
      // Check if file exists
      await fs.access(filePath);
      
      // Read and parse CSV file
      const fileContent = await fs.readFile(filePath, 'utf-8');
      
      // Simple CSV parsing
      const lines = fileContent.split('\n').filter(line => line.trim());
      if (lines.length < 2) {
        throw new Error('Invalid CSV file: no data found');
      }
      
      // Skip header line
      const dataLines = lines.slice(1);
      
      // Convert to map for easy lookup
      this.secrets = new Map();
      for (const line of dataLines) {
        const [service, login, password] = line.split(',').map(s => s.trim());
        if (service && login && password) {
          this.secrets.set(service.toLowerCase(), {
            login: login,
            password: password
          });
        }
      }
      
      return this.secrets;
    } catch (error) {
      if (error.code === 'ENOENT') {
        throw new Error(`Secrets file '${this.secretsFile}' not found in current directory`);
      }
      throw new Error(`Failed to load secrets: ${error.message}`);
    }
  }

  async call(input) {
    try {
      // Load secrets if not already loaded
      const secrets = await this._loadSecrets();
      
      if (input.operation === 'list') {
        // Return list of all service names
        const services = Array.from(secrets.keys());
        if (services.length === 0) {
          return "No credentials found in .secrets.csv";
        }
        return `Available services: ${services.join(', ')}`;
      }
      
      if (input.operation === 'read') {
        if (!input.service) {
          return "Error: Service name is required for 'read' operation";
        }
        
        const serviceName = input.service.toLowerCase();
        const credentials = secrets.get(serviceName);
        
        if (!credentials) {
          return `No credentials found for service: ${input.service}`;
        }
        
        // Return credentials in a structured format
        return JSON.stringify({
          service: input.service,
          login: credentials.login,
          password: credentials.password
        });
      }
      
      return "Invalid operation. Use 'list' or 'read <service>'";
      
    } catch (error) {
      return `SecretsManager error: ${error.message}`;
    }
  }
}

// For backwards compatibility
export default PlaywrightSecretsManager;