import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

/**
 * Google Search Tool - Dedicated tool for searching Google
 * Inspired by browser-use's search_google action
 */
export class PlaywrightSearchGoogleTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "search_google",
      "Search Google with a query. The query should be concrete and specific, like how humans search.",
      z.object({
        query: z.string().describe("The search query"),
        newTab: z.boolean().optional().default(false).describe("Open search in new tab")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ query, newTab = false }) {
    try {
      // Construct Google search URL with clean results (udm=14 removes ads/extra features)
      const searchUrl = `https://www.google.com/search?q=${encodeURIComponent(query)}&udm=14`;
      
      if (this.options.verbose) {
        console.log(`🔍 Searching Google for: "${query}"`);
      }

      let targetPage = this.page;
      
      // Smart tab management: reuse Google tab or create new one
      if (newTab) {
        // Create new tab for search
        const context = this.page.context();
        targetPage = await context.newPage();
        
        if (this.options.verbose) {
          console.log("📑 Opened new tab for search");
        }
      } else {
        // Check if we're already on Google to avoid unnecessary navigation
        const currentUrl = this.page.url();
        if (currentUrl.includes("google.com/search")) {
          // Already on Google search, just update query
          if (this.options.verbose) {
            console.log("♻️ Reusing existing Google search tab");
          }
        }
      }

      // Navigate to search URL
      await targetPage.goto(searchUrl, { 
        waitUntil: "domcontentloaded",
        timeout: 30000 
      });

      // Wait for search results to load
      try {
        // Wait for search results container
        await targetPage.waitForSelector("#search", { 
          timeout: 10000,
          state: "visible" 
        });
      } catch (e) {
        // Fallback: wait for any results indicator
        await targetPage.waitForSelector("[data-async-context]", {
          timeout: 5000,
          state: "visible"
        }).catch(() => {
          // Even if selectors fail, page might still have loaded
          console.warn("⚠️ Could not verify search results loaded, continuing anyway");
        });
      }

      // Brief pause to ensure results are rendered
      await targetPage.waitForTimeout(500);

      const resultMessage = newTab 
        ? `Searched for "${query}" in new tab`
        : `Searched for "${query}"`;

      if (this.options.verbose) {
        console.log(`${resultMessage}`);
      }

      // If new tab was created, update the page reference
      if (newTab) {
        this.page = targetPage;
      }

      return resultMessage;

    } catch (error) {
      const errorMsg = `Failed to search Google for "${query}": ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * DuckDuckGo Search Tool - Alternative privacy-focused search
 */
export class PlaywrightSearchDuckDuckGoTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "search_duckduckgo",
      "Search DuckDuckGo with a query (privacy-focused alternative to Google).",
      z.object({
        query: z.string().describe("The search query"),
        newTab: z.boolean().optional().default(false).describe("Open search in new tab")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ query, newTab = false }) {
    try {
      // Construct DuckDuckGo search URL
      const searchUrl = `https://duckduckgo.com/?q=${encodeURIComponent(query)}`;
      
      if (this.options.verbose) {
        console.log(`🦆 Searching DuckDuckGo for: "${query}"`);
      }

      let targetPage = this.page;
      
      if (newTab) {
        const context = this.page.context();
        targetPage = await context.newPage();
        
        if (this.options.verbose) {
          console.log("📑 Opened new tab for search");
        }
      }

      // Navigate to search URL
      await targetPage.goto(searchUrl, { 
        waitUntil: "domcontentloaded",
        timeout: 30000 
      });

      // Wait for search results to load
      try {
        await targetPage.waitForSelector(".results", { 
          timeout: 10000,
          state: "visible" 
        });
      } catch (e) {
        // Fallback selector
        await targetPage.waitForSelector("[data-testid='result']", {
          timeout: 5000,
          state: "visible"
        }).catch(() => {
          console.warn("⚠️ Could not verify search results loaded, continuing anyway");
        });
      }

      await targetPage.waitForTimeout(500);

      const resultMessage = newTab 
        ? `Searched DuckDuckGo for "${query}" in new tab`
        : `Searched DuckDuckGo for "${query}"`;

      if (this.options.verbose) {
        console.log(`${resultMessage}`);
      }

      if (newTab) {
        this.page = targetPage;
      }

      return resultMessage;

    } catch (error) {
      const errorMsg = `Failed to search DuckDuckGo for "${query}": ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}