import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

export class PlaywrightScrollTool extends BrowserTool {
  constructor(page) {
    super(
      "scroll_page",
      "Scroll the page up or down",
      z.object({
        direction: z.enum(["up", "down"]).describe("Scroll direction"),
        amount: z.number().optional().describe("Pixels to scroll (default: 500)"),
        reasoning: z.string().describe("Reasoning for this action")
      })
    );
    this.page = page;
  }

  async call({ direction, amount = 500, reasoning }) {
    try {
      const scrollAmount = direction === 'down' ? amount : -amount;
      await this.page.evaluate((pixels) => {
        window.scrollBy(0, pixels);
      }, scrollAmount);
      
      // Wait for any lazy-loaded content or dynamic changes
      try {
        await this.page.waitForLoadState('networkidle', { timeout: 2000 });
      } catch (timeoutError) {
        // If networkidle times out, wait a bit for content to load
        await this.page.waitForTimeout(500);
      }
      
      return `Scrolled ${direction} by ${amount}px`;
    } catch (error) {
      throw new Error(`Scroll failed: ${error.message}`);
    }
  }
}