import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

/**
 * Scroll to Text Tool - Find and scroll to specific text on the page
 * Inspired by browser-use's scroll_to_text action
 */
export class PlaywrightScrollToTextTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "scroll_to_text",
      "Find and scroll to specific text on the current page. Tries multiple strategies to locate the text.",
      z.object({
        text: z.string().describe("The text to find and scroll to"),
        exact: z.boolean()
          .optional()
          .default(false)
          .describe("Whether to match text exactly or allow partial matches"),
        timeout: z.number()
          .optional()
          .default(5000)
          .describe("Maximum time to wait for text to appear (milliseconds)")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ text, exact = false, timeout = 5000 }) {
    try {
      if (this.options.verbose) {
        console.log(`🔍 Searching for text: "${text}"${exact ? ' (exact match)' : ''}`);
      }

      // Try multiple locator strategies
      const locatorStrategies = [
        // Strategy 1: Built-in text locator
        () => this.page.getByText(text, { exact }),
        
        // Strategy 2: Text selector
        () => this.page.locator(`text=${exact ? `"${text}"` : text}`),
        
        // Strategy 3: XPath contains
        () => this.page.locator(`//*[contains(text(), '${text.replace(/'/g, "\\''")}')]`),
        
        // Strategy 4: Case-insensitive XPath (if not exact)
        () => !exact ? this.page.locator(
          `//*[contains(translate(text(), 'ABCDEFGHIJKLMNOPQRSTUVWXYZ', 'abcdefghijklmnopqrstuvwxyz'), '${text.toLowerCase().replace(/'/g, "\\''")}')]`
        ) : null,
        
        // Strategy 5: Check within common text containers
        () => this.page.locator(`p:has-text("${text}"), span:has-text("${text}"), div:has-text("${text}"), h1:has-text("${text}"), h2:has-text("${text}"), h3:has-text("${text}")`).first()
      ];

      let foundElement = null;
      let strategyUsed = "";

      // Try each strategy
      for (let i = 0; i < locatorStrategies.length; i++) {
        const strategy = locatorStrategies[i];
        const locator = strategy();
        
        if (!locator) continue;

        try {
          // Check if element exists and is visible
          const count = await locator.count();
          
          if (count > 0) {
            // Try to find a visible element
            for (let j = 0; j < Math.min(count, 10); j++) {
              const element = locator.nth(j);
              
              try {
                const isVisible = await element.isVisible({ timeout: 1000 });
                const box = await element.boundingBox({ timeout: 1000 });
                
                if (isVisible && box && box.width > 0 && box.height > 0) {
                  foundElement = element;
                  strategyUsed = `strategy ${i + 1}, element ${j + 1}/${count}`;
                  break;
                }
              } catch (e) {
                // Element might not be ready, continue to next
                continue;
              }
            }
            
            if (foundElement) break;
          }
        } catch (e) {
          // Strategy failed, try next one
          if (this.options.verbose) {
            console.log(`  Strategy ${i + 1} failed: ${e.message.substring(0, 50)}...`);
          }
          continue;
        }
      }

      // If no element found, try waiting for it to appear
      if (!foundElement && timeout > 0) {
        if (this.options.verbose) {
          console.log(`⏳ Text not immediately found, waiting up to ${timeout}ms...`);
        }

        try {
          const dynamicLocator = this.page.getByText(text, { exact });
          await dynamicLocator.first().waitFor({ 
            state: "visible", 
            timeout: timeout 
          });
          foundElement = dynamicLocator.first();
          strategyUsed = "dynamic wait";
        } catch (e) {
          // Still not found after waiting
        }
      }

      if (!foundElement) {
        const message = `Text "${text}" not found on the page${exact ? ' (exact match)' : ''}`;
        
        if (this.options.verbose) {
          console.log(`⚠️ ${message}`);
        }
        
        return message;
      }

      // Scroll the element into view
      try {
        await foundElement.scrollIntoViewIfNeeded({ timeout: 5000 });
        
        // Brief pause to let scroll animation complete
        await this.page.waitForTimeout(500);

        // Optional: highlight the element briefly
        try {
          await foundElement.evaluate(el => {
            const originalBorder = el.style.border;
            el.style.border = '3px solid red';
            setTimeout(() => {
              el.style.border = originalBorder;
            }, 2000);
          });
        } catch (e) {
          // Highlighting failed, not critical
        }

        const message = `Scrolled to text: "${text}" (found using ${strategyUsed})`;
        
        if (this.options.verbose) {
          console.log(`${message}`);
        }

        return message;

      } catch (scrollError) {
        // Element found but couldn't scroll to it
        const message = `Found text "${text}" but couldn't scroll to it: ${scrollError.message}`;
        
        if (this.options.verbose) {
          console.log(`⚠️ ${message}`);
        }
        
        return message;
      }

    } catch (error) {
      const errorMsg = `Failed to scroll to text "${text}": ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * Scroll to Element Tool - Scroll to element by selector
 */
export class PlaywrightScrollToElementTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "scroll_to_element",
      "Scroll to an element using a CSS selector or XPath.",
      z.object({
        selector: z.string().describe("CSS selector or XPath to find the element"),
        timeout: z.number()
          .optional()
          .default(5000)
          .describe("Maximum time to wait for element (milliseconds)")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ selector, timeout = 5000 }) {
    try {
      if (this.options.verbose) {
        console.log(`🔍 Looking for element: ${selector}`);
      }

      // Create locator
      const element = this.page.locator(selector).first();

      // Wait for element to be visible
      try {
        await element.waitFor({ state: "visible", timeout });
      } catch (e) {
        const message = `Element "${selector}" not found or not visible`;
        
        if (this.options.verbose) {
          console.log(`⚠️ ${message}`);
        }
        
        return message;
      }

      // Scroll into view
      await element.scrollIntoViewIfNeeded();
      await this.page.waitForTimeout(500);

      const message = `Scrolled to element: ${selector}`;
      
      if (this.options.verbose) {
        console.log(`${message}`);
      }

      return message;

    } catch (error) {
      const errorMsg = `Failed to scroll to element "${selector}": ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}