import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { PopupHelper } from "./popup-helper.js";

export class PlaywrightNavigateTool extends BrowserTool {
  constructor(page) {
    super(
      "navigate_to_url",
      "Navigate the browser to a specific URL",
      z.object({
        url: z.string().describe("URL to navigate to (with or without protocol)"),
        reasoning: z.string().describe("Reasoning for this navigation action")
      })
    );
    this.page = page;
  }

  async call({ url, reasoning }) {
    try {
      // Reset popup injection state for new pages - this prevents rate limiting
      // caused by failed popup injections on previous pages
      PopupHelper.resetPopupState();

      // Normalize URL by adding protocol if missing
      let normalizedUrl = url;
      if (!url.match(/^(https?|file):\/\//i)) {
        // No protocol specified - default to https
        normalizedUrl = `https://${url}`;
      }
      // file:// URLs are passed through as-is

      await this.page.goto(normalizedUrl, {
        waitUntil: 'domcontentloaded',
        timeout: 30000
      });

      // Additionally wait for network to settle, but don't fail if it times out
      try {
        await this.page.waitForLoadState('networkidle', { timeout: 5000 });
      } catch (e) {
        // Network might not settle on some pages like LinkedIn, that's OK
        console.log('[NavigateTool] Network did not settle, continuing...');
      }
      return `Successfully navigated to ${normalizedUrl}`;
    } catch (error) {
      throw new Error(`Navigation failed: ${error.message}`);
    }
  }
}
