// Indexed Interaction Tools for Playwright - Browser-use inspired approach
// These tools work with element indices from get_page_content

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { PopupHelper } from "./popup-helper.js";
import path from "path";
import fs from "fs/promises";

// Helper function to get screenshot directory
async function getScreenshotDir() {
  // Check if we're running in test mode with a specific directory
  const testDir = process.env.VIBE_TEST_DIR;
  if (testDir) {
    // Ensure the directory exists
    await fs.mkdir(testDir, { recursive: true });
    return testDir;
  }
  // Default to current directory
  return process.cwd();
}

// Tool to click by index
export class PlaywrightClickByIndexTool extends BrowserTool {
  constructor(page) {
    super(
      "click_by_index",
      "Click an element using its index number from indexed elements. Always use get_indexed_elements or get_page_content first to see available elements with scores. Use openInNewTab=true to open links in a new tab.",
      z.object({
        index: z.number().describe("The index number of the element to click (from [index:score] format)"),
        reasoning: z.string().describe("Reasoning for this action"),
        openInNewTab: z.boolean().nullable().optional().describe("If true and element is a link, open in new tab instead of navigating current tab")
      })
    );
    this.page = page;
  }

  async call({ index, reasoning, openInNewTab }) {
    try {
      // Show reasoning popup
      await PopupHelper.showReasoningPopup(this.page, reasoning, 'Click Element');

      // First, get fresh indexed elements by evaluating the page
      const elements = await this.page.evaluate(() => {
        // This should match the indexed content extractor logic
        const allElements = document.querySelectorAll('a, button, input, select, textarea, [role="button"], [role="link"], [onclick], [data-action]');
        const indexed = [];
        
        allElements.forEach((el, idx) => {
          const rect = el.getBoundingClientRect();
          if (rect.width > 0 && rect.height > 0) {
            indexed.push({
              index: idx,
              element: el,
              selector: el.tagName.toLowerCase() + 
                        (el.id ? `#${el.id}` : '') +
                        (el.className ? `.${el.className.split(' ').join('.')}` : '')
            });
          }
        });
        
        return indexed;
      });

      // More lenient validation - just check if index is negative
      // The actual element count will be validated when trying to click
      if (index < 0) {
        throw new Error(`Index ${index} is invalid. Index must be >= 0`);
      }

      // Warn if index seems high but don't fail yet
      if (index >= elements.length) {
        console.warn(`Index ${index} may be out of range. Found ${elements.length} visible elements. Will attempt to click anyway.`);
      }

      // Click the element at the specified index using cached elements if available
      const clickResult = await this.page.evaluate(({ idx, newTab }) => {
        // Try to use cached elements first
        let el = null;
        if (window.ElementCache) {
          const cacheId = window.ElementCache.getLatestCacheId();
          if (cacheId) {
            el = window.ElementCache.getElement(cacheId, idx);
          }
        }

        // If no cached element, fall back to re-querying
        if (!el) {
            const interactiveSelectors = [
              'a[href]',
              'button',
              'input',
              'select',
              'textarea',
              '[role="button"]',
              '[role="link"]',
              '[role="checkbox"]',
              '[role="radio"]',
              '[role="combobox"]',
              '[role="listbox"]',
              '[role="option"]',
              '[role="menuitem"]',
              '[role="tab"]',
              '[onclick]',
              '[tabindex]:not([tabindex="-1"]):not(c-wiz)',
              '[contenteditable="true"]',
              '.autocomplete-suggestion',
              '.dropdown-item',
              '[data-suggestion]',
              'li[role="option"]',
              'div[role="option"]'
            ];

            const allElements = document.querySelectorAll(interactiveSelectors.join(', '));

            // Just get the element by index from all matched elements
            // Don't filter by visibility - trust the index from content extractor
            if (idx >= 0 && idx < allElements.length) {
              el = allElements[idx];
            }
        }

        if (!el) {
          throw new Error(`Could not find element at index ${idx}`);
        }

        // Scroll to element with smooth animation and center it in view
        el.scrollIntoView({
          behavior: 'smooth',
          block: 'center',
          inline: 'center'
        });

        // Additional scroll adjustment to ensure element is truly visible
        // Check if element is in viewport after initial scroll
        setTimeout(() => {
          const rect = el.getBoundingClientRect();
          const isInViewport = (
            rect.top >= 0 &&
            rect.left >= 0 &&
            rect.bottom <= window.innerHeight &&
            rect.right <= window.innerWidth
          );

          // If element is still not fully in viewport, scroll again
          if (!isInViewport) {
            window.scrollBy({
              top: rect.top - window.innerHeight / 2,
              left: rect.left - window.innerWidth / 2,
              behavior: 'smooth'
            });
          }
        }, 500);

        // Handle openInNewTab for links
        if (newTab) {
          // Check if element is a link or has an href
          const href = el.href || el.getAttribute('href');
          if (href && href !== '#' && !href.startsWith('javascript:')) {
            // Open in new tab using window.open
            window.open(href, '_blank');
            return { openedInNewTab: true, href };
          }
          // If not a link, fall back to regular click
        }

        // Wait a bit for scroll to complete before clicking
        setTimeout(() => {
          el.click();
        }, 600);

        return { openedInNewTab: false };
      }, { idx: index, newTab: openInNewTab || false });

      // Wait for scrolling animation and click to complete
      await this.page.waitForTimeout(700);

      // Enhanced wait for potential navigation/reload
      try {
        // Wait for navigation if it happens (e.g., after cookie consent)
        await Promise.race([
          this.page.waitForLoadState('networkidle', { timeout: 5000 }),
          this.page.waitForLoadState('domcontentloaded', { timeout: 5000 })
        ]);
        
        // Additional wait for dynamic content to stabilize
        await this.page.waitForTimeout(1000);
      } catch (waitError) {
        // If wait times out, continue anyway
        console.debug('[click_by_index] Wait after click timed out, continuing...');
      }
      
      // Highlight the clicked element and take screenshot as proof
      try {
        // Highlight the clicked element with a bright border
        await this.page.evaluate((idx) => {
          let el = null;
          if (window.ElementCache) {
            const cacheId = window.ElementCache.getLatestCacheId();
            if (cacheId) {
              el = window.ElementCache.getElement(cacheId, idx);
            }
          }

          if (!el) {
            const interactiveSelectors = [
              'a[href]', 'button', 'input', 'select', 'textarea',
              '[role="button"]', '[role="link"]', '[role="checkbox"]',
              '[role="radio"]', '[role="combobox"]', '[role="listbox"]',
              '[role="option"]', '[role="menuitem"]', '[role="tab"]',
              '[onclick]', '[tabindex]:not([tabindex="-1"]):not(c-wiz)',
              '[contenteditable="true"]', '.autocomplete-suggestion',
              '.dropdown-item', '[data-suggestion]',
              'li[role="option"]', 'div[role="option"]'
            ];
            const allElements = document.querySelectorAll(interactiveSelectors.join(', '));
            if (idx >= 0 && idx < allElements.length) {
              el = allElements[idx];
            }
          }

          if (el) {
            // Store original styles
            el.dataset.originalBorder = el.style.border || '';
            el.dataset.originalBoxShadow = el.style.boxShadow || '';
            el.dataset.originalOutline = el.style.outline || '';

            // Apply highlight styles
            el.style.border = '4px solid #00ff00';
            el.style.boxShadow = '0 0 20px #00ff00, inset 0 0 20px rgba(0,255,0,0.3)';
            el.style.outline = '2px dashed #ff0000';

            // Add a label showing what was clicked
            const label = document.createElement('div');
            label.innerHTML = `CLICKED: Index ${idx}`;
            label.style.cssText = 'position: fixed; top: 10px; right: 10px; background: #00ff00; color: black; padding: 10px; font-size: 20px; font-weight: bold; z-index: 999999; border: 3px solid black;';
            label.id = 'click-proof-label';
            document.body.appendChild(label);
          }
        }, index);

        // Wait for highlight to be visible
        await this.page.waitForTimeout(200);

        // Take the screenshot with highlights
        const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
        const screenshotDir = await getScreenshotDir();
        const screenshotPath = path.join(screenshotDir, `click-proof-${timestamp}.png`);
        await this.page.screenshot({ path: screenshotPath, fullPage: false });
        console.log(`[Screenshot] Click proof saved: ${path.basename(screenshotPath)}`);

        // Remove highlights after screenshot
        await this.page.evaluate((idx) => {
          // Remove label
          const label = document.getElementById('click-proof-label');
          if (label) label.remove();

          // Restore original styles
          let el = null;
          if (window.ElementCache) {
            const cacheId = window.ElementCache.getLatestCacheId();
            if (cacheId) {
              el = window.ElementCache.getElement(cacheId, idx);
            }
          }

          if (!el) {
            const interactiveSelectors = [
              'a[href]', 'button', 'input', 'select', 'textarea',
              '[role="button"]', '[role="link"]', '[role="checkbox"]',
              '[role="radio"]', '[role="combobox"]', '[role="listbox"]',
              '[role="option"]', '[role="menuitem"]', '[role="tab"]',
              '[onclick]', '[tabindex]:not([tabindex="-1"]):not(c-wiz)',
              '[contenteditable="true"]', '.autocomplete-suggestion',
              '.dropdown-item', '[data-suggestion]',
              'li[role="option"]', 'div[role="option"]'
            ];
            const allElements = document.querySelectorAll(interactiveSelectors.join(', '));
            if (idx >= 0 && idx < allElements.length) {
              el = allElements[idx];
            }
          }

          if (el && el.dataset.originalBorder !== undefined) {
            el.style.border = el.dataset.originalBorder;
            el.style.boxShadow = el.dataset.originalBoxShadow;
            el.style.outline = el.dataset.originalOutline;
            delete el.dataset.originalBorder;
            delete el.dataset.originalBoxShadow;
            delete el.dataset.originalOutline;
          }
        }, index);

      } catch (screenshotError) {
        console.error('Failed to take click screenshot:', screenshotError.message);
      }

      if (clickResult && clickResult.openedInNewTab) {
        return `Opened link in new tab: ${clickResult.href}`;
      }
      return `Clicked element at index ${index}`;
    } catch (error) {
      throw new Error(`Click by index failed: ${error.message}`);
    }
  }
}

// Tool to fill by index
export class PlaywrightFillByIndexTool extends BrowserTool {
  constructor(page) {
    super(
      "fill_by_index",
      "Fill a form field using its index number from indexed elements. Use higher scored elements for better results.",
      z.object({
        index: z.number().describe("The index number of the field to fill (from [index:score] format)"),
        value: z.string().describe("The value to fill into the field"),
        reasoning: z.string().describe("Reasoning for this action")
      })
    );
    this.page = page;
  }

  async call({ index, value, reasoning }) {
    try {
      // Show reasoning popup
      await PopupHelper.showReasoningPopup(this.page, reasoning, 'Fill Field');

      // Fill the element at the specified index using cached elements if available
      await this.page.evaluate(({ idx, val }) => {
        // Helper function to simulate realistic typing
        function simulateTyping(element, text) {
          // Focus the element
          element.focus();
          
          // Clear existing content more thoroughly
          if (element.tagName.toLowerCase() === 'input' || element.tagName.toLowerCase() === 'textarea') {
            // Select all text
            element.select && element.select();
            
            // Clear via multiple methods for reliability
            element.value = '';
            
            // Dispatch clear events
            element.dispatchEvent(new Event('input', { bubbles: true }));
            element.dispatchEvent(new Event('change', { bubbles: true }));
          } else if (element.contentEditable === 'true') {
            // For contenteditable elements
            const range = document.createRange();
            range.selectNodeContents(element);
            const selection = window.getSelection();
            selection.removeAllRanges();
            selection.addRange(range);
            document.execCommand('delete', false, null);
          }
          
          // Type character by character
          for (let i = 0; i < text.length; i++) {
            const char = text[i];
            
            // Dispatch keydown
            const keydownEvent = new KeyboardEvent('keydown', {
              key: char,
              code: `Key${char.toUpperCase()}`,
              charCode: char.charCodeAt(0),
              keyCode: char.charCodeAt(0),
              which: char.charCodeAt(0),
              bubbles: true,
              cancelable: true
            });
            element.dispatchEvent(keydownEvent);
            
            // Actually set the value
            if (element.tagName.toLowerCase() === 'input' || element.tagName.toLowerCase() === 'textarea') {
              element.value += char;
            } else if (element.contentEditable === 'true') {
              document.execCommand('insertText', false, char);
            }
            
            // Dispatch keypress (deprecated but some sites still use it)
            const keypressEvent = new KeyboardEvent('keypress', {
              key: char,
              code: `Key${char.toUpperCase()}`,
              charCode: char.charCodeAt(0),
              keyCode: char.charCodeAt(0),
              which: char.charCodeAt(0),
              bubbles: true,
              cancelable: true
            });
            element.dispatchEvent(keypressEvent);
            
            // Dispatch input event after each character
            element.dispatchEvent(new InputEvent('input', {
              data: char,
              inputType: 'insertText',
              bubbles: true,
              cancelable: true
            }));
            
            // Dispatch keyup
            const keyupEvent = new KeyboardEvent('keyup', {
              key: char,
              code: `Key${char.toUpperCase()}`,
              charCode: char.charCodeAt(0),
              keyCode: char.charCodeAt(0),
              which: char.charCodeAt(0),
              bubbles: true,
              cancelable: true
            });
            element.dispatchEvent(keyupEvent);
          }
          
          // Final change event
          element.dispatchEvent(new Event('change', { bubbles: true }));
          
          // Dispatch Enter key to submit if it's a search/filter field
          const enterEvent = new KeyboardEvent('keydown', {
            key: 'Enter',
            code: 'Enter',
            keyCode: 13,
            which: 13,
            bubbles: true,
            cancelable: true
          });
          element.dispatchEvent(enterEvent);
          
          // Also dispatch keyup for Enter
          const enterUpEvent = new KeyboardEvent('keyup', {
            key: 'Enter',
            code: 'Enter',
            keyCode: 13,
            which: 13,
            bubbles: true,
            cancelable: true
          });
          element.dispatchEvent(enterUpEvent);
          
          // Blur and refocus to trigger any validation
          element.blur();
          element.focus();
        }
        
        // Try to use cached elements first
        let el = null;
        if (window.ElementCache) {
          const cacheId = window.ElementCache.getLatestCacheId();
          if (cacheId) {
            el = window.ElementCache.getElement(cacheId, idx);
          }
        }
        
        // If no cached element, fall back to re-querying
        if (!el) {
            const interactiveSelectors = [
              'a[href]',
              'button',
              'input',
              'select',
              'textarea',
              '[role="button"]',
              '[role="link"]',
              '[role="checkbox"]',
              '[role="radio"]',
              '[role="combobox"]',
              '[role="listbox"]',
              '[role="option"]',
              '[role="menuitem"]',
              '[role="tab"]',
              '[onclick]',
              '[tabindex]:not([tabindex="-1"]):not(c-wiz)',
              '[contenteditable="true"]',
              '.autocomplete-suggestion',
              '.dropdown-item',
              '[data-suggestion]',
              'li[role="option"]',
              'div[role="option"]'
            ];

            const allElements = document.querySelectorAll(interactiveSelectors.join(', '));

            // Just get the element by index from all matched elements
            // Don't filter by visibility - trust the index from content extractor
            if (idx >= 0 && idx < allElements.length) {
              el = allElements[idx];
            }
        }
        
        if (!el) {
          throw new Error(`Could not find element at index ${idx}`);
        }
        
        // Check if element is fillable
        const tagName = el.tagName.toLowerCase();
        const isFillable = tagName === 'input' || tagName === 'textarea' || 
                          tagName === 'select' || el.contentEditable === 'true';
        
        if (!isFillable) {
          throw new Error(`Element at index ${idx} is not fillable (${tagName})`);
        }

        // Scroll to element with smooth animation and center it in view
        el.scrollIntoView({
          behavior: 'smooth',
          block: 'center',
          inline: 'center'
        });

        // Additional scroll adjustment to ensure element is truly visible
        // Check if element is in viewport after initial scroll
        setTimeout(() => {
          const rect = el.getBoundingClientRect();
          const isInViewport = (
            rect.top >= 0 &&
            rect.left >= 0 &&
            rect.bottom <= window.innerHeight &&
            rect.right <= window.innerWidth
          );

          // If element is still not fully in viewport, scroll again
          if (!isInViewport) {
            window.scrollBy({
              top: rect.top - window.innerHeight / 2,
              left: rect.left - window.innerWidth / 2,
              behavior: 'smooth'
            });
          }
        }, 500);

        if (tagName === 'select') {
          // For select elements, just set the value directly
          el.focus();
          el.value = val;
          el.dispatchEvent(new Event('change', { bubbles: true }));
        } else {
          // For input, textarea, and contenteditable, use realistic typing
          simulateTyping(el, val);
        }
      }, { idx: index, val: value });

      // Wait for scrolling animation to complete
      await this.page.waitForTimeout(800);

      // Small delay to let any UI updates complete
      await this.page.waitForTimeout(100);

      // Highlight the filled element and take screenshot as proof
      try {
        // Highlight the filled element with a bright border and show the value
        await this.page.evaluate(({ idx, val }) => {
          let el = null;
          if (window.ElementCache) {
            const cacheId = window.ElementCache.getLatestCacheId();
            if (cacheId) {
              el = window.ElementCache.getElement(cacheId, idx);
            }
          }

          if (!el) {
            const interactiveSelectors = [
              'a[href]', 'button', 'input', 'select', 'textarea',
              '[role="button"]', '[role="link"]', '[role="checkbox"]',
              '[role="radio"]', '[role="combobox"]', '[role="listbox"]',
              '[role="option"]', '[role="menuitem"]', '[role="tab"]',
              '[onclick]', '[tabindex]:not([tabindex="-1"]):not(c-wiz)',
              '[contenteditable="true"]', '.autocomplete-suggestion',
              '.dropdown-item', '[data-suggestion]',
              'li[role="option"]', 'div[role="option"]'
            ];
            const allElements = document.querySelectorAll(interactiveSelectors.join(', '));
            if (idx >= 0 && idx < allElements.length) {
              el = allElements[idx];
            }
          }

          if (el) {
            // Store original styles
            el.dataset.originalBorder = el.style.border || '';
            el.dataset.originalBoxShadow = el.style.boxShadow || '';
            el.dataset.originalOutline = el.style.outline || '';
            el.dataset.originalBackground = el.style.background || '';

            // Apply highlight styles
            el.style.border = '4px solid #0080ff';
            el.style.boxShadow = '0 0 20px #0080ff, inset 0 0 20px rgba(0,128,255,0.3)';
            el.style.outline = '2px dashed #ffff00';
            el.style.background = 'rgba(0,128,255,0.1)';

            // Add a label showing what was filled
            const label = document.createElement('div');
            label.innerHTML = `FILLED: Index ${idx}<br>VALUE: "${val}"`;
            label.style.cssText = 'position: fixed; top: 10px; right: 10px; background: #0080ff; color: white; padding: 15px; font-size: 18px; font-weight: bold; z-index: 999999; border: 3px solid white; max-width: 400px; word-wrap: break-word;';
            label.id = 'fill-proof-label';
            document.body.appendChild(label);

            // Also add an arrow pointing to the element
            const arrow = document.createElement('div');
            const rect = el.getBoundingClientRect();
            arrow.innerHTML = '⬅ FILLED HERE';
            arrow.style.cssText = `position: absolute; left: ${rect.right + 10}px; top: ${rect.top + window.scrollY}px; background: #ff0000; color: white; padding: 5px 10px; font-size: 16px; font-weight: bold; z-index: 999998; border-radius: 5px;`;
            arrow.id = 'fill-arrow-label';
            document.body.appendChild(arrow);
          }
        }, { idx: index, val: value });

        // Wait for highlight to be visible
        await this.page.waitForTimeout(300);

        // Take the screenshot with highlights
        const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
        const screenshotDir = await getScreenshotDir();
        const screenshotPath = path.join(screenshotDir, `fill-proof-${timestamp}.png`);
        await this.page.screenshot({ path: screenshotPath, fullPage: false });
        console.log(`[Screenshot] Fill proof saved: ${path.basename(screenshotPath)}`);

        // Remove highlights after screenshot
        await this.page.evaluate((idx) => {
          // Remove labels
          const label = document.getElementById('fill-proof-label');
          if (label) label.remove();
          const arrow = document.getElementById('fill-arrow-label');
          if (arrow) arrow.remove();

          // Restore original styles
          let el = null;
          if (window.ElementCache) {
            const cacheId = window.ElementCache.getLatestCacheId();
            if (cacheId) {
              el = window.ElementCache.getElement(cacheId, idx);
            }
          }

          if (!el) {
            const interactiveSelectors = [
              'a[href]', 'button', 'input', 'select', 'textarea',
              '[role="button"]', '[role="link"]', '[role="checkbox"]',
              '[role="radio"]', '[role="combobox"]', '[role="listbox"]',
              '[role="option"]', '[role="menuitem"]', '[role="tab"]',
              '[onclick]', '[tabindex]:not([tabindex="-1"]):not(c-wiz)',
              '[contenteditable="true"]', '.autocomplete-suggestion',
              '.dropdown-item', '[data-suggestion]',
              'li[role="option"]', 'div[role="option"]'
            ];
            const allElements = document.querySelectorAll(interactiveSelectors.join(', '));
            if (idx >= 0 && idx < allElements.length) {
              el = allElements[idx];
            }
          }

          if (el && el.dataset.originalBorder !== undefined) {
            el.style.border = el.dataset.originalBorder;
            el.style.boxShadow = el.dataset.originalBoxShadow;
            el.style.outline = el.dataset.originalOutline;
            el.style.background = el.dataset.originalBackground;
            delete el.dataset.originalBorder;
            delete el.dataset.originalBoxShadow;
            delete el.dataset.originalOutline;
            delete el.dataset.originalBackground;
          }
        }, index);

      } catch (screenshotError) {
        console.error('Failed to take fill screenshot:', screenshotError.message);
      }

      return `Filled element at index ${index} with value: ${value}`;
    } catch (error) {
      throw new Error(`Fill by index failed: ${error.message}`);
    }
  }
}