import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { PopupHelper } from "./popup-helper.js";

/**
 * Go Back Tool - Navigate back in browser history
 * Inspired by browser-use's go_back action
 */
export class PlaywrightGoBackTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "go_back",
      "Navigate back to the previous page in browser history.",
      z.object({
        waitForLoad: z.boolean()
          .optional()
          .default(true)
          .describe("Wait for the page to load after going back")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ waitForLoad = true }) {
    try {
      // Reset popup injection state for new pages - prevents rate limiting
      PopupHelper.resetPopupState();

      // Check if we can go back
      const currentUrl = this.page.url();

      if (this.options.verbose) {
        console.log(`🔙 Navigating back from: ${currentUrl}`);
      }

      // Attempt to go back
      const response = await this.page.goBack({
        waitUntil: waitForLoad ? "domcontentloaded" : "commit",
        timeout: 30000
      });

      if (!response) {
        // No history to go back to
        const message = "No previous page in history to go back to";

        if (this.options.verbose) {
          console.log(`ℹ️ ${message}`);
        }

        return message;
      }

      const newUrl = this.page.url();
      const message = `Navigated back from ${currentUrl} to ${newUrl}`;

      if (this.options.verbose) {
        console.log(`${message}`);
      }

      return message;

    } catch (error) {
      const errorMsg = `Failed to go back: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * Go Forward Tool - Navigate forward in browser history
 */
export class PlaywrightGoForwardTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "go_forward",
      "Navigate forward to the next page in browser history.",
      z.object({
        waitForLoad: z.boolean()
          .optional()
          .default(true)
          .describe("Wait for the page to load after going forward")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ waitForLoad = true }) {
    try {
      // Reset popup injection state for new pages - prevents rate limiting
      PopupHelper.resetPopupState();

      const currentUrl = this.page.url();

      if (this.options.verbose) {
        console.log(`⏩ Navigating forward from: ${currentUrl}`);
      }

      // Attempt to go forward
      const response = await this.page.goForward({
        waitUntil: waitForLoad ? "domcontentloaded" : "commit",
        timeout: 30000
      });

      if (!response) {
        const message = "No forward page in history to navigate to";

        if (this.options.verbose) {
          console.log(`ℹ️ ${message}`);
        }

        return message;
      }

      const newUrl = this.page.url();
      const message = `Navigated forward from ${currentUrl} to ${newUrl}`;

      if (this.options.verbose) {
        console.log(`${message}`);
      }

      return message;

    } catch (error) {
      const errorMsg = `Failed to go forward: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * Reload Tool - Reload the current page
 */
export class PlaywrightReloadTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "reload_page",
      "Reload/refresh the current page.",
      z.object({
        hardReload: z.boolean()
          .optional()
          .default(false)
          .describe("Perform a hard reload (bypass cache)")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ hardReload = false }) {
    try {
      // Reset popup injection state for reloaded pages - prevents rate limiting
      PopupHelper.resetPopupState();

      const currentUrl = this.page.url();

      if (this.options.verbose) {
        console.log(`🔄 Reloading page: ${currentUrl}${hardReload ? ' (hard reload)' : ''}`);
      }

      // Reload the page
      await this.page.reload({
        waitUntil: "domcontentloaded",
        timeout: 30000
      });

      // If hard reload requested, also clear cache
      if (hardReload) {
        // Force clear cache by evaluating JavaScript
        await this.page.evaluate(() => {
          if ('caches' in window) {
            caches.keys().then(names => {
              names.forEach(name => caches.delete(name));
            });
          }
        });
      }

      const message = hardReload
        ? `Hard reloaded page: ${currentUrl}`
        : `Reloaded page: ${currentUrl}`;

      if (this.options.verbose) {
        console.log(`${message}`);
      }

      return message;

    } catch (error) {
      const errorMsg = `Failed to reload page: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}
