import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

export class PlaywrightFillTool extends BrowserTool {
  constructor(page) {
    super(
      "fill",
      "Fill a form field with text",
      z.object({
        selector: z.union([
          z.string().describe("CSS selector or role-based locator"),
          z.object({
            role: z.string().optional(),
            text: z.string().optional(),
            name: z.string().optional(),
            placeholder: z.string().optional(),
            title: z.string().optional(),
            selector: z.string().optional()
          }).describe("Locator object with role, text, or other attributes")
        ]).optional(),
        locator: z.object({
          role: z.string().optional(),
          text: z.string().optional(),
          name: z.string().optional(),
          placeholder: z.string().optional(),
          title: z.string().optional(),
          selector: z.string().optional()
        }).optional().describe("Alternative locator object format"),
        value: z.string().describe("Value to fill"),
        reasoning: z.string().describe("Reasoning for this action")
      })
    );
    this.page = page;
  }

  async call({ selector, locator, value, reasoning }) {
    try {
      // Handle locator object format
      if (locator && typeof locator === 'object') {
        selector = locator;
      }
      
      // Get element based on selector type
      let element;
      
      // Handle object-based locator
      if (typeof selector === 'object') {
        if (selector.role) {
          const options = {};
          if (selector.text) options.name = selector.text;
          if (selector.name) options.name = selector.name;
          element = this.page.getByRole(selector.role, options);
        } else if (selector.text) {
          element = this.page.getByText(selector.text);
        } else if (selector.placeholder) {
          element = this.page.getByPlaceholder(selector.placeholder);
        } else if (selector.title) {
          element = this.page.getByTitle(selector.title);
        } else if (selector.selector) {
          element = this.page.locator(selector.selector);
        } else {
          throw new Error("Invalid locator object: must have role, text, placeholder, title, or selector property");
        }
      }
      // Handle string selector
      else if (typeof selector === 'string') {
        element = this.page.locator(selector);
      } else {
        throw new Error("Selector must be a string or locator object");
      }
      const count = await element.count();
      
      if (count === 0) {
        throw new Error(`No input field found matching selector: ${selector}. Ensure you're targeting an input, textarea, or contenteditable element.`);
      } else if (count > 1) {
        throw new Error(`Selector "${selector}" matched ${count} elements. Use a more specific selector for the input field.`);
      }

      // Check if element is visible and enabled
      const isVisible = await element.isVisible();
      const isEnabled = await element.isEnabled();
      
      if (!isVisible) {
        throw new Error(`Input field ${selector} is not visible. Wait for it to appear or check if it's hidden behind another element.`);
      }
      if (!isEnabled) {
        throw new Error(`Input field ${selector} is disabled. Cannot fill disabled fields.`);
      }

      // Use fill on the element
      await element.fill(value);
      
      // Wait for any form submission or dynamic content changes
      try {
        await this.page.waitForLoadState('networkidle', { timeout: 3000 });
      } catch (timeoutError) {
        // If networkidle times out, at least wait for domcontentloaded
        await this.page.waitForLoadState('domcontentloaded', { timeout: 1000 }).catch(() => {});
      }
      
      // Create readable selector description
      const selectorDesc = typeof selector === 'object' 
        ? JSON.stringify(selector)
        : selector;
      
      return `Filled ${selectorDesc} with "${value}"`;
    } catch (error) {
      // Enhance error messages
      if (error.message.includes('not an <input>')) {
        throw new Error(`Fill failed - selector doesn't point to a fillable element. ${error.message}. Use this tool only for input fields, textareas, or contenteditable elements.`);
      }
      throw error;
    }
  }
}