import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import fs from "fs/promises";
import path from "path";
// PDF generation requires: npm install marked pdfkit
// import { marked } from "marked";
// import PDFDocument from "pdfkit";
import { createWriteStream } from "fs";

/**
 * Write File Tool - Write content to various file formats
 * Inspired by browser-use's write_file action
 */
export class PlaywrightWriteFileTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "write_file",
      "Write or append content to a file. Supports .md, .txt, .json, .csv formats. For .pdf, provide markdown content.",
      z.object({
        fileName: z.string().describe("Name of the file to write"),
        content: z.string().describe("Content to write to the file"),
        append: z.boolean()
          .optional()
          .default(false)
          .describe("Append to existing file instead of overwriting"),
        directory: z.string()
          .optional()
          .default("./output")
          .describe("Directory to save the file in")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ fileName, content, append = false, directory = "./output" }) {
    try {
      // Ensure directory exists
      const dirPath = path.resolve(directory);
      await fs.mkdir(dirPath, { recursive: true });
      
      // Create full file path
      const filePath = path.join(dirPath, fileName);
      const ext = path.extname(fileName).toLowerCase();
      
      if (this.options.verbose) {
        console.log(`💾 ${append ? 'Appending to' : 'Writing'} file: ${filePath}`);
      }

      // Handle different file types
      switch (ext) {
        case '.json':
          await this.writeJSON(filePath, content, append);
          break;
        
        case '.csv':
          await this.writeCSV(filePath, content, append);
          break;
        
        case '.pdf':
          // PDF generation disabled - requires: npm install marked pdfkit
          throw new Error("PDF generation not available. Install 'marked' and 'pdfkit' packages to enable.");
          // if (append) {
          //   throw new Error("Cannot append to PDF files");
          // }
          // await this.writePDF(filePath, content);
          break;
        
        case '.md':
        case '.txt':
        case '.log':
        default:
          // Plain text files
          if (append) {
            await fs.appendFile(filePath, content + '\n', 'utf-8');
          } else {
            await fs.writeFile(filePath, content, 'utf-8');
          }
          break;
      }

      // Get file size
      const stats = await fs.stat(filePath);
      const sizeKB = (stats.size / 1024).toFixed(2);
      
      const message = `${append ? 'Appended to' : 'Wrote'} ${fileName} (${sizeKB} KB) at ${filePath}`;
      
      if (this.options.verbose) {
        console.log(`   ${message}`);
      }

      return message;

    } catch (error) {
      const errorMsg = `Failed to write file: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }

  async writeJSON(filePath, content, append) {
    try {
      // Try to parse as JSON
      const newData = JSON.parse(content);
      
      if (append) {
        // Read existing file if it exists
        let existingData = [];
        try {
          const existingContent = await fs.readFile(filePath, 'utf-8');
          existingData = JSON.parse(existingContent);
          if (!Array.isArray(existingData)) {
            existingData = [existingData];
          }
        } catch (e) {
          // File doesn't exist or isn't valid JSON
        }
        
        // Append new data
        if (Array.isArray(newData)) {
          existingData.push(...newData);
        } else {
          existingData.push(newData);
        }
        
        await fs.writeFile(filePath, JSON.stringify(existingData, null, 2), 'utf-8');
      } else {
        await fs.writeFile(filePath, JSON.stringify(newData, null, 2), 'utf-8');
      }
    } catch (e) {
      // If not valid JSON, write as plain text
      if (append) {
        await fs.appendFile(filePath, content + '\n', 'utf-8');
      } else {
        await fs.writeFile(filePath, content, 'utf-8');
      }
    }
  }

  async writeCSV(filePath, content, append) {
    // Simple CSV handling - assume content is already CSV formatted
    if (append) {
      // Remove header row if appending
      const lines = content.split('\n');
      if (lines.length > 1 && lines[0].includes(',')) {
        // Skip first line if it looks like a header
        content = lines.slice(1).join('\n');
      }
      await fs.appendFile(filePath, '\n' + content, 'utf-8');
    } else {
      await fs.writeFile(filePath, content, 'utf-8');
    }
  }

  // PDF generation disabled - requires: npm install marked pdfkit
  // async writePDF(filePath, markdownContent) {
  //   return new Promise((resolve, reject) => {
  //     try {
  //       const html = marked(markdownContent);
  //       const text = html.replace(/<[^>]*>/g, '');
  //       const doc = new PDFDocument();
  //       const stream = createWriteStream(filePath);
  //       doc.pipe(stream);
  //       doc.fontSize(12);
  //       doc.text(text, 50, 50);
  //       doc.end();
  //       stream.on('finish', resolve);
  //       stream.on('error', reject);
  //     } catch (error) {
  //       reject(error);
  //     }
  //   });
  // }
}

/**
 * Read File Tool - Read content from files
 * Inspired by browser-use's read_file action
 */
export class PlaywrightReadFileTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "read_file",
      "Read content from a file",
      z.object({
        fileName: z.string().describe("Name or path of the file to read"),
        maxLines: z.number()
          .optional()
          .default(1000)
          .describe("Maximum number of lines to read"),
        encoding: z.string()
          .optional()
          .default("utf-8")
          .describe("File encoding")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ fileName, maxLines = 1000, encoding = "utf-8" }) {
    try {
      const filePath = path.resolve(fileName);
      
      if (this.options.verbose) {
        console.log(`📖 Reading file: ${filePath}`);
      }

      // Check if file exists
      const stats = await fs.stat(filePath);
      if (!stats.isFile()) {
        throw new Error(`Path is not a file: ${filePath}`);
      }

      // Read file content
      const content = await fs.readFile(filePath, encoding);
      
      // Limit lines if needed
      const lines = content.split('\n');
      let result = content;
      let truncated = false;
      
      if (lines.length > maxLines) {
        result = lines.slice(0, maxLines).join('\n');
        truncated = true;
      }

      const sizeKB = (stats.size / 1024).toFixed(2);
      const message = `Read ${filePath} (${sizeKB} KB, ${lines.length} lines${truncated ? `, showing first ${maxLines}` : ''})`;
      
      if (this.options.verbose) {
        console.log(`   ${message}`);
      }

      return `${message}\n\nContent:\n${result}`;

    } catch (error) {
      const errorMsg = `Failed to read file: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * Replace String in File Tool - Replace text in files
 * Inspired by browser-use's replace_file_str action
 */
export class PlaywrightReplaceInFileTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "replace_in_file",
      "Replace occurrences of a string in a file",
      z.object({
        fileName: z.string().describe("Name or path of the file"),
        oldString: z.string().describe("String to replace"),
        newString: z.string().describe("Replacement string"),
        replaceAll: z.boolean()
          .optional()
          .default(true)
          .describe("Replace all occurrences or just the first")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ fileName, oldString, newString, replaceAll = true }) {
    try {
      const filePath = path.resolve(fileName);
      
      if (this.options.verbose) {
        console.log(`🔄 Replacing in file: ${filePath}`);
        console.log(`   Old: "${oldString.substring(0, 50)}${oldString.length > 50 ? '...' : ''}"`);
        console.log(`   New: "${newString.substring(0, 50)}${newString.length > 50 ? '...' : ''}"`);
      }

      // Read file
      const content = await fs.readFile(filePath, 'utf-8');
      
      // Count occurrences
      const occurrences = content.split(oldString).length - 1;
      
      if (occurrences === 0) {
        return `No occurrences of "${oldString}" found in ${fileName}`;
      }

      // Replace
      let newContent;
      if (replaceAll) {
        newContent = content.replaceAll(oldString, newString);
      } else {
        newContent = content.replace(oldString, newString);
      }

      // Write back
      await fs.writeFile(filePath, newContent, 'utf-8');

      const replacedCount = replaceAll ? occurrences : 1;
      const message = `Replaced ${replacedCount} occurrence${replacedCount > 1 ? 's' : ''} in ${fileName}`;
      
      if (this.options.verbose) {
        console.log(`   ${message}`);
      }

      return message;

    } catch (error) {
      const errorMsg = `Failed to replace in file: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * List Files Tool - List files in a directory
 */
export class PlaywrightListFilesTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "list_files",
      "List files in a directory",
      z.object({
        directory: z.string()
          .optional()
          .default(".")
          .describe("Directory to list files from"),
        pattern: z.string()
          .optional()
          .describe("File name pattern to filter (e.g., '*.txt')"),
        recursive: z.boolean()
          .optional()
          .default(false)
          .describe("List files recursively in subdirectories")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ directory = ".", pattern, recursive = false }) {
    try {
      const dirPath = path.resolve(directory);
      
      if (this.options.verbose) {
        console.log(`📁 Listing files in: ${dirPath}`);
      }

      const files = await this.listFiles(dirPath, pattern, recursive);
      
      const message = `Found ${files.length} file${files.length !== 1 ? 's' : ''} in ${dirPath}`;
      
      if (this.options.verbose) {
        console.log(`   ${message}`);
        if (files.length <= 20) {
          files.forEach(f => console.log(`   - ${f}`));
        } else {
          console.log(`   First 10: ${files.slice(0, 10).join(', ')}`);
          console.log(`   ...and ${files.length - 10} more`);
        }
      }

      return `${message}\n\nFiles:\n${files.join('\n')}`;

    } catch (error) {
      const errorMsg = `Failed to list files: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }

  async listFiles(dirPath, pattern, recursive, basePath = '') {
    const entries = await fs.readdir(dirPath, { withFileTypes: true });
    const files = [];

    for (const entry of entries) {
      const fullPath = path.join(dirPath, entry.name);
      const relativePath = path.join(basePath, entry.name);
      
      if (entry.isDirectory() && recursive) {
        const subFiles = await this.listFiles(fullPath, pattern, recursive, relativePath);
        files.push(...subFiles);
      } else if (entry.isFile()) {
        if (!pattern || this.matchPattern(entry.name, pattern)) {
          files.push(relativePath || entry.name);
        }
      }
    }

    return files;
  }

  matchPattern(fileName, pattern) {
    // Simple glob pattern matching
    const regex = pattern
      .replace(/\./g, '\\.')
      .replace(/\*/g, '.*')
      .replace(/\?/g, '.');
    return new RegExp(`^${regex}$`).test(fileName);
  }
}