import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

/**
 * Enhanced Send Keys Tool - Advanced keyboard input with better error handling
 * Inspired by browser-use's send_keys action
 */
export class PlaywrightEnhancedSendKeysTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "send_keys",
      "Send special keys or keyboard shortcuts. Supports complex combinations like Control+Shift+T, function keys, and special keys like Escape, Enter, Tab, etc.",
      z.object({
        keys: z.union([
          z.string(),
          z.array(z.string())
        ]).describe("Key(s) to press - can be a single key/combo or array of keys"),
        delay: z.number()
          .optional()
          .default(50)
          .describe("Delay between key presses in milliseconds (for sequences)"),
        holdDelay: z.number()
          .optional()
          .default(0)
          .describe("How long to hold the key(s) down in milliseconds")
      })
    );
    this.page = page;
    this.options = options;
    
    // Map common key aliases to Playwright key names
    this.keyMap = {
      // Common aliases
      "Ctrl": "Control",
      "Cmd": "Meta",
      "Command": "Meta",
      "Win": "Meta",
      "Windows": "Meta",
      "Option": "Alt",
      "Return": "Enter",
      "Esc": "Escape",
      "Del": "Delete",
      "Ins": "Insert",
      "PgUp": "PageUp",
      "PgDn": "PageDown",
      
      // Arrow keys
      "Up": "ArrowUp",
      "Down": "ArrowDown",
      "Left": "ArrowLeft",
      "Right": "ArrowRight",
      
      // Special characters that need mapping
      "Plus": "+",
      "Minus": "-",
      "Equals": "=",
      "Comma": ",",
      "Period": ".",
      "Slash": "/",
      "Backslash": "\\",
      "Semicolon": ";",
      "Quote": "'",
      "BracketLeft": "[",
      "BracketRight": "]",
      
      // Function keys are handled as-is (F1-F12)
    };
  }

  normalizeKey(key) {
    // Check if it's a known alias
    if (this.keyMap[key]) {
      return this.keyMap[key];
    }
    
    // Handle function keys
    if (/^F\d{1,2}$/.test(key)) {
      return key; // F1-F12 are valid as-is
    }
    
    // Handle single letters/numbers
    if (key.length === 1) {
      return key;
    }
    
    // Handle modifier combinations (e.g., "Control+C")
    if (key.includes("+")) {
      const parts = key.split("+");
      return parts.map(part => this.normalizeKey(part.trim())).join("+");
    }
    
    // Return as-is and let Playwright handle it
    return key;
  }

  async call({ keys, delay = 50, holdDelay = 0 }) {
    try {
      // Convert to array if single string
      const keySequence = Array.isArray(keys) ? keys : [keys];
      
      if (this.options.verbose) {
        console.log(`⌨️ Sending keys: ${JSON.stringify(keySequence)}`);
      }

      const results = [];
      
      for (let i = 0; i < keySequence.length; i++) {
        const rawKey = keySequence[i];
        const normalizedKey = this.normalizeKey(rawKey);
        
        if (this.options.verbose && normalizedKey !== rawKey) {
          console.log(`   Normalized "${rawKey}" to "${normalizedKey}"`);
        }

        try {
          // Try to press the key
          if (holdDelay > 0) {
            // Hold the key down for specified duration
            await this.page.keyboard.down(normalizedKey);
            await this.page.waitForTimeout(holdDelay);
            await this.page.keyboard.up(normalizedKey);
          } else {
            // Normal key press
            await this.page.keyboard.press(normalizedKey);
          }
          
          results.push(`✓ ${rawKey}`);
          
          // Add delay between keys if in sequence
          if (i < keySequence.length - 1 && delay > 0) {
            await this.page.waitForTimeout(delay);
          }
          
        } catch (keyError) {
          // If single key fails, try breaking it down
          if (keyError.message.includes("Unknown key") && normalizedKey.includes("+")) {
            if (this.options.verbose) {
              console.log(`   Complex combo failed, trying sequential approach...`);
            }
            
            // Try pressing keys sequentially for complex combos
            const parts = normalizedKey.split("+");
            const modifiers = [];
            let mainKey = "";
            
            // Separate modifiers from main key
            for (const part of parts) {
              if (["Control", "Alt", "Shift", "Meta"].includes(part)) {
                modifiers.push(part);
              } else {
                mainKey = part;
              }
            }
            
            try {
              // Press modifiers down
              for (const mod of modifiers) {
                await this.page.keyboard.down(mod);
              }
              
              // Press main key
              if (mainKey) {
                await this.page.keyboard.press(mainKey);
              }
              
              // Release modifiers
              for (const mod of modifiers.reverse()) {
                await this.page.keyboard.up(mod);
              }
              
              results.push(`✓ ${rawKey} (sequential)`);
            } catch (seqError) {
              // If even sequential fails, try individual keys
              if (this.options.verbose) {
                console.log(`   Sequential failed, trying individual keys...`);
              }
              
              for (const part of parts) {
                try {
                  await this.page.keyboard.press(part);
                  await this.page.waitForTimeout(10);
                } catch (e) {
                  console.warn(`   ⚠️ Failed to press "${part}": ${e.message}`);
                  results.push(`✗ ${part}`);
                }
              }
              results.push(`⚠️ ${rawKey} (partial)`);
            }
          } else {
            // Single key genuinely failed
            console.warn(`   ⚠️ Failed to press "${rawKey}": ${keyError.message}`);
            results.push(`✗ ${rawKey}`);
            
            // Don't throw for individual key failures in a sequence
            if (keySequence.length === 1) {
              throw keyError;
            }
          }
        }
      }

      // Wait for any triggered actions to complete
      try {
        await this.page.waitForLoadState('networkidle', { timeout: 2000 });
      } catch (e) {
        // Network idle timeout is not critical
        await this.page.waitForTimeout(300);
      }

      const summary = results.join(", ");
      const message = `Sent keys: ${summary}`;
      
      if (this.options.verbose) {
        console.log(`   ${message}`);
      }

      return message;

    } catch (error) {
      const errorMsg = `Failed to send keys: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * Type Text Tool - Type text character by character (for fields that don't accept paste)
 */
export class PlaywrightTypeTextTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "type_text",
      "Type text character by character with optional delay. Useful for fields that don't accept paste or need slow typing.",
      z.object({
        text: z.string().describe("Text to type"),
        delay: z.number()
          .optional()
          .default(50)
          .describe("Delay between characters in milliseconds"),
        selector: z.string()
          .optional()
          .describe("Optional selector to focus before typing")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ text, delay = 50, selector }) {
    try {
      // Focus element if selector provided
      if (selector) {
        const element = await this.page.locator(selector).first();
        await element.focus();
        
        if (this.options.verbose) {
          console.log(`   Focused element: ${selector}`);
        }
      }

      // Type text with delay
      await this.page.keyboard.type(text, { delay });

      return `Typed ${text.length} characters${selector ? ` into ${selector}` : ''}`;

    } catch (error) {
      const errorMsg = `Failed to type text: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}

/**
 * Keyboard Combo Tool - Press multiple keys simultaneously
 */
export class PlaywrightKeyboardComboTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "keyboard_combo",
      "Press multiple keys simultaneously (e.g., for selecting all text with Ctrl+A)",
      z.object({
        modifiers: z.array(z.enum(["Control", "Alt", "Shift", "Meta"]))
          .optional()
          .default([])
          .describe("Modifier keys to hold"),
        key: z.string().describe("Main key to press while holding modifiers")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ modifiers = [], key }) {
    try {
      // Hold down modifiers
      for (const mod of modifiers) {
        await this.page.keyboard.down(mod);
      }

      // Press main key
      await this.page.keyboard.press(key);

      // Release modifiers in reverse order
      for (const mod of modifiers.reverse()) {
        await this.page.keyboard.up(mod);
      }

      const combo = modifiers.length > 0 
        ? `${modifiers.join("+")}+${key}`
        : key;

      return `Pressed keyboard combo: ${combo}`;

    } catch (error) {
      // Make sure to release any held keys on error
      try {
        await this.page.keyboard.up("Control");
        await this.page.keyboard.up("Alt");
        await this.page.keyboard.up("Shift");
        await this.page.keyboard.up("Meta");
      } catch (e) {
        // Ignore cleanup errors
      }
      
      const errorMsg = `Failed to press keyboard combo: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}