import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { search } from 'duck-duck-scrape';

/**
 * DuckDuckGo API Search Tool - Uses free DuckDuckGo search without browser
 * This is more reliable than browser-based search as it doesn't trigger bot detection
 * Based on LangChain's DuckDuckGoSearch implementation
 */
export class PlaywrightDuckDuckGoAPISearchTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "search_duckduckgo_api",
      "Search DuckDuckGo using their free API (no browser needed, no bot detection)",
      z.object({
        query: z.string().describe("The search query"),
        maxResults: z.number().optional().default(5).describe("Maximum number of results to return")
      })
    );
    this.page = page; // Keep for compatibility, but not used
    this.options = options;
  }

  async call({ query, maxResults = 5 }) {
    try {
      if (this.options.verbose) {
        console.log(`🦆 Searching DuckDuckGo API for: "${query}"`);
      }

      // Use duck-duck-scrape to search
      const searchResults = await search(query, {
        safeSearch: 0, // 0 = off, 1 = moderate, 2 = strict
        locale: 'en-us'
      });

      if (!searchResults || !searchResults.results || searchResults.results.length === 0) {
        throw new Error('No search results found');
      }

      // Format results
      const results = searchResults.results.slice(0, maxResults).map((result, index) => ({
        position: index + 1,
        title: result.title || '',
        snippet: result.description || '',
        link: result.url || ''
      }));

      if (this.options.verbose) {
        console.log(`Found ${results.length} results`);
      }

      // Format output message
      let message = `DuckDuckGo search for "${query}" returned ${results.length} results:\n\n`;
      
      results.forEach(result => {
        message += `${result.position}. ${result.title}\n`;
        message += `   ${result.link}\n`;
        if (result.snippet) {
          message += `   ${result.snippet.substring(0, 150)}...\n`;
        }
        message += '\n';
      });

      return message;

    } catch (error) {
      const errorMsg = `DuckDuckGo API search failed for "${query}": ${error.message}`;
      if (this.options.verbose) {
        console.error(`❌ ${errorMsg}`);
      }
      throw new Error(errorMsg);
    }
  }
}

/**
 * Alternative using fetch directly (in case duck-duck-scrape has issues)
 */
export class PlaywrightDuckDuckGoHTMLSearchTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "search_duckduckgo_html",
      "Search DuckDuckGo by fetching HTML directly (fallback method)",
      z.object({
        query: z.string().describe("The search query"),
        maxResults: z.number().optional().default(5).describe("Maximum number of results to return")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ query, maxResults = 5 }) {
    try {
      if (this.options.verbose) {
        console.log(`🦆 Searching DuckDuckGo HTML for: "${query}"`);
      }

      // DuckDuckGo HTML endpoint
      const url = `https://html.duckduckgo.com/html/?q=${encodeURIComponent(query)}`;
      
      // Fetch HTML
      const response = await fetch(url, {
        headers: {
          'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        }
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const html = await response.text();

      // Simple regex-based extraction (not perfect but works)
      const results = [];
      const resultPattern = /<a class="result__a"[^>]*href="([^"]*)"[^>]*>(.*?)<\/a>.*?<a class="result__snippet"[^>]*>(.*?)<\/a>/gs;
      
      let match;
      let count = 0;
      while ((match = resultPattern.exec(html)) !== null && count < maxResults) {
        const [, url, title, snippet] = match;
        results.push({
          position: count + 1,
          title: this.cleanHTML(title),
          snippet: this.cleanHTML(snippet),
          link: url
        });
        count++;
      }

      if (results.length === 0) {
        // Try alternative pattern
        const altPattern = /<h2[^>]*>.*?<a[^>]*href="([^"]*)"[^>]*>(.*?)<\/a>/gs;
        while ((match = altPattern.exec(html)) !== null && count < maxResults) {
          const [, url, title] = match;
          results.push({
            position: count + 1,
            title: this.cleanHTML(title),
            snippet: '',
            link: url
          });
          count++;
        }
      }

      if (results.length === 0) {
        throw new Error('Could not parse search results from HTML');
      }

      if (this.options.verbose) {
        console.log(`Found ${results.length} results`);
      }

      // Format output message
      let message = `DuckDuckGo search for "${query}" returned ${results.length} results:\n\n`;
      
      results.forEach(result => {
        message += `${result.position}. ${result.title}\n`;
        message += `   ${result.link}\n`;
        if (result.snippet) {
          message += `   ${result.snippet.substring(0, 150)}...\n`;
        }
        message += '\n';
      });

      return message;

    } catch (error) {
      const errorMsg = `DuckDuckGo HTML search failed for "${query}": ${error.message}`;
      if (this.options.verbose) {
        console.error(`❌ ${errorMsg}`);
      }
      throw new Error(errorMsg);
    }
  }

  cleanHTML(text) {
    return text
      .replace(/<[^>]*>/g, '') // Remove HTML tags
      .replace(/&amp;/g, '&')
      .replace(/&lt;/g, '<')
      .replace(/&gt;/g, '>')
      .replace(/&quot;/g, '"')
      .replace(/&#39;/g, "'")
      .replace(/\s+/g, ' ')
      .trim();
  }
}