import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

/**
 * Get Dropdown Options Tool - Extract all options from dropdowns and menus
 * Inspired by browser-use's get_dropdown_options action
 */
export class PlaywrightGetDropdownOptionsTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "get_dropdown_options",
      "Get all available options from a dropdown, select element, or ARIA menu",
      z.object({
        selector: z.string().describe("CSS selector for the dropdown/select element")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ selector }) {
    try {
      if (this.options.verbose) {
        console.log(`📋 Getting dropdown options for: ${selector}`);
      }

      const element = this.page.locator(selector).first();
      
      // Check if element exists
      const exists = await element.count() > 0;
      if (!exists) {
        throw new Error(`Element not found: ${selector}`);
      }

      // Get element type
      const tagName = await element.evaluate(el => el.tagName.toLowerCase());
      const role = await element.getAttribute('role');
      
      let options = [];
      
      if (tagName === 'select') {
        // Native select element
        options = await this.getSelectOptions(element);
      } else if (role === 'combobox' || role === 'listbox' || role === 'menu') {
        // ARIA dropdown/menu
        options = await this.getAriaOptions(element, role);
      } else {
        // Try common dropdown patterns
        options = await this.getCustomDropdownOptions(element);
      }

      if (options.length === 0) {
        // Fallback: try to click and find options
        options = await this.getOptionsByClicking(element);
      }

      const message = `Found ${options.length} options in ${tagName}${role ? `[role="${role}"]` : ''}`;
      
      if (this.options.verbose) {
        console.log(`   ${message}`);
        if (options.length <= 10) {
          options.forEach((opt, i) => {
            console.log(`   ${i + 1}. ${opt.text}${opt.value !== opt.text ? ` (value: ${opt.value})` : ''}`);
          });
        } else {
          console.log(`   First 5: ${options.slice(0, 5).map(o => o.text).join(', ')}`);
          console.log(`   Last 5: ${options.slice(-5).map(o => o.text).join(', ')}`);
        }
      }

      return {
        message,
        options,
        type: tagName,
        role: role || undefined
      };

    } catch (error) {
      const errorMsg = `Failed to get dropdown options: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }

  async getSelectOptions(selectElement) {
    return await selectElement.evaluate(select => {
      const options = Array.from(select.options);
      return options.map(option => ({
        text: option.textContent?.trim() || '',
        value: option.value,
        selected: option.selected,
        disabled: option.disabled,
        index: option.index
      }));
    });
  }

  async getAriaOptions(element, role) {
    // First try to find options without clicking
    let optionElements = await this.findAriaOptions(element);
    
    if (optionElements.length === 0) {
      // Try clicking to open the dropdown
      await element.click();
      await this.page.waitForTimeout(500);
      optionElements = await this.findAriaOptions(element);
    }

    const options = [];
    for (const optElement of optionElements) {
      const text = await optElement.textContent();
      const value = await optElement.getAttribute('data-value') || 
                    await optElement.getAttribute('value') || 
                    text;
      const selected = await optElement.getAttribute('aria-selected') === 'true';
      const disabled = await optElement.getAttribute('aria-disabled') === 'true';
      
      options.push({
        text: text?.trim() || '',
        value: value?.trim() || '',
        selected,
        disabled
      });
    }

    return options;
  }

  async findAriaOptions(element) {
    // Look for options in various ARIA patterns
    const selectors = [
      '[role="option"]',
      '[role="menuitem"]',
      '[role="menuitemradio"]',
      '[role="menuitemcheckbox"]',
      'li[role="option"]',
      'div[role="option"]',
      'span[role="option"]'
    ];

    for (const selector of selectors) {
      // First check within the element
      let options = await element.locator(selector).all();
      if (options.length > 0) return options;

      // Then check in associated listbox
      const ariaControls = await element.getAttribute('aria-controls');
      if (ariaControls) {
        const listbox = this.page.locator(`#${ariaControls}`);
        options = await listbox.locator(selector).all();
        if (options.length > 0) return options;
      }

      // Check siblings
      const parent = element.locator('..');
      options = await parent.locator(selector).all();
      if (options.length > 0) return options;
    }

    return [];
  }

  async getCustomDropdownOptions(element) {
    // Common patterns for custom dropdowns
    const patterns = [
      { container: element, options: 'option' },
      { container: element, options: 'li' },
      { container: element, options: '.option' },
      { container: element, options: '.dropdown-item' },
      { container: element, options: '.menu-item' },
      { container: element, options: '[data-value]' }
    ];

    for (const pattern of patterns) {
      const optionElements = await pattern.container.locator(pattern.options).all();
      if (optionElements.length > 0) {
        const options = [];
        for (const optElement of optionElements) {
          const text = await optElement.textContent();
          const value = await optElement.getAttribute('data-value') || 
                        await optElement.getAttribute('value') || 
                        text;
          
          options.push({
            text: text?.trim() || '',
            value: value?.trim() || ''
          });
        }
        return options;
      }
    }

    return [];
  }

  async getOptionsByClicking(element) {
    try {
      // Click to open dropdown
      await element.click();
      await this.page.waitForTimeout(500);

      // Look for newly visible elements that might be options
      const possibleSelectors = [
        '.dropdown-menu:visible li',
        '.menu:visible li',
        '.options:visible > *',
        '[class*="dropdown"]:visible [class*="option"]',
        '[class*="menu"]:visible [class*="item"]'
      ];

      for (const selector of possibleSelectors) {
        const optionElements = await this.page.locator(selector).all();
        if (optionElements.length > 0) {
          const options = [];
          for (const optElement of optionElements) {
            const text = await optElement.textContent();
            const value = await optElement.getAttribute('data-value') || text;
            
            options.push({
              text: text?.trim() || '',
              value: value?.trim() || ''
            });
          }

          // Close dropdown
          await this.page.keyboard.press('Escape');
          
          return options;
        }
      }

      // Close dropdown if nothing found
      await this.page.keyboard.press('Escape');
      
    } catch (e) {
      // Ignore click errors
    }

    return [];
  }
}

/**
 * Select Dropdown Option Tool - Select an option from a dropdown
 */
export class PlaywrightSelectDropdownOptionTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "select_dropdown_option",
      "Select an option from a dropdown by text or value",
      z.object({
        selector: z.string().describe("CSS selector for the dropdown/select element"),
        option: z.union([
          z.string(),
          z.number()
        ]).describe("Option to select (text, value, or index)"),
        by: z.enum(["text", "value", "index"])
          .optional()
          .default("text")
          .describe("How to match the option")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ selector, option, by = "text" }) {
    try {
      const element = this.page.locator(selector).first();
      const tagName = await element.evaluate(el => el.tagName.toLowerCase());

      if (tagName === 'select') {
        // Native select element
        const optionStr = option ? String(option) : '';
        if (by === "text") {
          await element.selectOption({ label: optionStr });
        } else if (by === "value") {
          await element.selectOption({ value: optionStr });
        } else if (by === "index") {
          await element.selectOption({ index: Number(option) });
        }
      } else {
        // Custom dropdown - click to open, then click option
        await element.click();
        await this.page.waitForTimeout(300);

        let optionLocator;
        if (by === "text") {
          optionLocator = this.page.locator(`text="${option}"`).first();
        } else if (by === "value") {
          optionLocator = this.page.locator(`[data-value="${option}"], [value="${option}"]`).first();
        } else {
          // For index, get all options and click the nth one
          const options = await this.page.locator('[role="option"], li, .option, .dropdown-item').all();
          if (options[Number(option)]) {
            await options[Number(option)].click();
            return `Selected option at index ${option}`;
          } else {
            throw new Error(`No option at index ${option}`);
          }
        }

        await optionLocator.click();
      }

      return `Selected option: ${option} (by ${by})`;

    } catch (error) {
      const errorMsg = `Failed to select dropdown option: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}