import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

export class PlaywrightClickTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "click",
      "Click an element on the page. When multiple elements match (e.g., multiple 'Sign in' buttons), the tool automatically selects the best one based on visibility, enabled state, form context, and element type. Prefer using indexed elements [index:score] when available.",
      z.object({
        selector: z.union([
          z.string().describe("CSS selector, text content, or role-based locator"),
          z.object({
            role: z.string().optional(),
            text: z.string().optional(),
            name: z.string().optional(),
            placeholder: z.string().optional(),
            title: z.string().optional(),
            selector: z.string().optional()
          }).describe("Locator object with role, text, or other attributes")
        ]).optional(),
        locator: z.object({
          role: z.string().optional(),
          text: z.string().optional(),
          name: z.string().optional(),
          placeholder: z.string().optional(),
          title: z.string().optional(),
          selector: z.string().optional()
        }).optional().describe("Alternative locator object format"),
        index: z.number().optional().describe("Index of element to click when multiple match (0-based)"),
        reasoning: z.string().describe("Reasoning for this click action")
      })
    );
    this.page = page;
    this.verbose = options.verbose || false;
  }

  async call({ selector, locator, index = 0, reasoning }) {
    try {
      // Handle locator object format
      if (locator && typeof locator === 'object') {
        selector = locator;
      }
      
      // Try different locator strategies
      let element;
      
      // Handle object-based locator
      if (typeof selector === 'object') {
        if (selector.role) {
          const options = {};
          if (selector.text) options.name = selector.text;
          if (selector.name) options.name = selector.name;
          element = this.page.getByRole(selector.role, options);
        } else if (selector.text) {
          element = this.page.getByText(selector.text);
        } else if (selector.placeholder) {
          element = this.page.getByPlaceholder(selector.placeholder);
        } else if (selector.title) {
          element = this.page.getByTitle(selector.title);
        } else if (selector.selector) {
          element = this.page.locator(selector.selector);
        } else {
          throw new Error("Invalid locator object: must have role, text, placeholder, title, or selector property");
        }
      }
      // Handle string selector
      else if (typeof selector === 'string') {
        // Try as text first
        if (selector.startsWith('text:')) {
          const text = selector.substring(5);
          element = this.page.getByText(text);
        }
        // Try as role
        else if (selector.startsWith('role:')) {
          const roleMatch = selector.match(/role:(\w+)(?:\[(.+)\])?/);
          if (roleMatch) {
            const [, role, attributes] = roleMatch;
            const options = {};
            if (attributes) {
              // Parse attributes like name=Submit
              const attrMatch = attributes.match(/(\w+)=(.+)/);
              if (attrMatch) {
                options[attrMatch[1]] = attrMatch[2];
              }
            }
            element = this.page.getByRole(role, options);
          }
        }
        // Default to CSS selector
        else {
          element = this.page.locator(selector);
        }
      } else {
        throw new Error("Selector must be a string or locator object");
      }

      // Check element count before clicking
      const count = await element.count();
      if (count === 0) {
        throw new Error(`No elements found matching selector: ${typeof selector === 'object' ? JSON.stringify(selector) : selector}. Try using find_elements tool first to locate the correct selector.`);
      } else if (count > 1 && index >= count) {
        throw new Error(`Selector matched ${count} elements but index ${index} is out of range. Valid indices are 0 to ${count - 1}.`);
      } else if (count > 1) {
        // When multiple elements match, try to pick the best one
        let selectedIndex = index;
        
        // If clicking a button and no explicit index was provided (index is 0 by default)
        if (index === 0 && (selector?.role === 'button' || selector?.text?.toLowerCase().includes('sign in') || selector?.name?.toLowerCase().includes('sign in'))) {
          // Try to find the most likely submit button
          const elements = [];
          for (let i = 0; i < count; i++) {
            const el = element.nth(i);
            try {
              const isVisible = await el.isVisible();
              const isEnabled = await el.isEnabled();
              const tag = await el.evaluate(e => e.tagName.toLowerCase());
              const type = await el.evaluate(e => e.getAttribute('type'));
              const form = await el.evaluate(e => !!e.closest('form'));
              
              elements.push({
                index: i,
                visible: isVisible,
                enabled: isEnabled,
                tag: tag,
                type: type,
                inForm: form
              });
            } catch (e) {
              // Skip elements that can't be evaluated
            }
          }
          
          // Prefer: visible, enabled, button/input[type=submit] inside a form
          const scoredElements = elements.map(el => ({
            ...el,
            score: (el.visible ? 10 : 0) + 
                   (el.enabled ? 10 : 0) + 
                   (el.inForm ? 20 : 0) +
                   (el.tag === 'button' ? 5 : 0) +
                   (el.type === 'submit' ? 15 : 0)
          }));
          
          // Sort by score and pick the best one
          scoredElements.sort((a, b) => b.score - a.score);
          
          // Log all scored elements in verbose mode
          if (this.verbose) {
            console.log(`\n📊 Scoring ${count} matching elements:`);
            scoredElements.forEach((el, idx) => {
              console.log(`  [${el.index}] Score: ${el.score} | Visible: ${el.visible} | Enabled: ${el.enabled} | InForm: ${el.inForm} | Tag: ${el.tag} | Type: ${el.type || 'none'}`);
            });
          }
          
          if (scoredElements.length > 0 && scoredElements[0].score > 0) {
            selectedIndex = scoredElements[0].index;
            console.log(`\nSelected element at index ${selectedIndex} with highest score: ${scoredElements[0].score}`);
          }
        }
        
        // Use the selected index
        element = element.nth(selectedIndex);
        if (!this.verbose) {
          console.log(`Multiple elements (${count}) matched, clicking element at index ${selectedIndex}`);
        }
      }

      // Ensure element is actually clickable (not covered by overlays)
      await element.waitFor({ state: 'visible' });
      
      // Try to click
      try {
        await element.click();
      } catch (clickError) {
        // If click failed due to interception, provide specific error
        if (clickError.message.includes('intercepts pointer events')) {
          // Extract what's blocking
          const blockingElement = clickError.message.match(/<([^>]+)>/)?.[1] || 'unknown element';
          throw new Error(`Click blocked by overlay: ${blockingElement}. Try waiting for the overlay to disappear or use JavaScript click.`);
        }
        throw clickError;
      }
      
      // Wait for any changes after click
      await this.page.waitForLoadState('domcontentloaded');
      
      // Create readable selector description
      const selectorDesc = typeof selector === 'object' 
        ? JSON.stringify(selector)
        : selector;
      
      return `Clicked element: ${selectorDesc}${count > 1 ? ` (index ${selectedIndex} of ${count} matches)` : ''}`;
    } catch (error) {
      // Enhance error messages with recovery suggestions
      if (error.message.includes('strict mode violation')) {
        throw new Error(`Click failed - multiple elements found. ${error.message}. Use find_elements tool to identify the correct element.`);
      } else if (error.message.includes('Timeout')) {
        throw new Error(`Click failed - element not found or not clickable. ${error.message}. Ensure the page is loaded and element is visible.`);
      }
      throw error;
    }
  }
}