// Tool for handling autocomplete and dropdown interactions
import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

export class PlaywrightAutocompleteTool extends BrowserTool {
  constructor(page) {
    super(
      "type_and_select",
      "Type text into a field and select from autocomplete dropdown. Use this for search fields with suggestions.",
      z.object({
        text: z.string().describe("Text to type into the field"),
        selectText: z.string().optional().describe("Text to look for in dropdown options (if different from typed text)"),
        waitTime: z.number().optional().default(2000).describe("Time to wait for dropdown in milliseconds"),
        reasoning: z.string().describe("Reasoning for this action")
      })
    );
    this.page = page;
  }

  async call({ text, selectText, waitTime, reasoning }) {
    try {
      // Type the text with realistic delays
      for (const char of text) {
        await this.page.keyboard.type(char);
        // Random delay between 50-150ms for human-like typing
        const delay = Math.floor(Math.random() * 100) + 50;
        await this.page.waitForTimeout(delay);
      }
      
      // Wait for dropdown to appear
      await this.page.waitForTimeout(waitTime);
      
      // Look for dropdown options
      const dropdownFound = await this.selectFromDropdown(selectText || text);
      
      if (dropdownFound) {
        return `Typed "${text}" and selected from dropdown successfully`;
      } else {
        // No dropdown found, just press Enter to accept the typed text
        await this.page.keyboard.press('Enter');
        return `Typed "${text}" and pressed Enter (no dropdown found)`;
      }
      
    } catch (error) {
      throw new Error(`Autocomplete interaction failed: ${error.message}`);
    }
  }
  
  async selectFromDropdown(searchText) {
    try {
      // Common selectors for dropdown options
      const dropdownSelectors = [
        '[role="option"]',
        '[role="listitem"]',
        '.autocomplete-suggestion',
        '.dropdown-item',
        '.suggestion',
        '.search-suggestion',
        'li[data-suggestion]',
        '.pac-item', // Google Places autocomplete
        '.gws-flights-autocomplete-suggestion', // Google Flights specific
        '[data-test*="suggestion"]',
        '[data-testid*="suggestion"]'
      ];
      
      // Try each selector
      for (const selector of dropdownSelectors) {
        const options = await this.page.$$(selector);
        
        if (options.length > 0) {
          // Found dropdown options
          
          // Look for the best match
          for (const option of options) {
            const optionText = await option.textContent();
            
            // Check if this option contains our search text
            if (optionText && optionText.toLowerCase().includes(searchText.toLowerCase())) {
              // Selecting matching option
              await option.click();
              await this.page.waitForTimeout(500);
              return true;
            }
          }
          
          // If no exact match, click the first option
          await options[0].click();
          await this.page.waitForTimeout(500);
          return true;
        }
      }
      
      // Also check if dropdown appeared but with different structure
      const visibleElements = await this.page.evaluate(() => {
        const elements = Array.from(document.querySelectorAll('*'));
        const dropdownKeywords = ['suggestion', 'autocomplete', 'dropdown', 'option', 'result'];
        
        return elements.filter(el => {
          const isVisible = el.offsetWidth > 0 && el.offsetHeight > 0;
          if (!isVisible) return false;
          
          const classes = el.className.toLowerCase();
          const id = el.id.toLowerCase();
          
          return dropdownKeywords.some(keyword => 
            classes.includes(keyword) || id.includes(keyword)
          );
        }).length;
      });
      
      // Potential dropdown elements detected but not interactable
      
      return false;
      
    } catch (error) {
      // Error checking for dropdown
      return false;
    }
  }
}