import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

export class PlaywrightHoverTool extends BrowserTool {
  constructor(page) {
    super(
      "hover_element",
      "Hover over an element to trigger tooltips or dropdowns",
      z.object({
        selector: z.string().describe("CSS selector of element to hover over"),
        reasoning: z.string().describe("Reasoning for this action")
      })
    );
    this.page = page;
  }

  async call({ selector, reasoning }) {
    try {
      await this.page.hover(selector);
      await this.page.waitForTimeout(500); // Wait for hover effects
      return `Hovered over element: ${selector}`;
    } catch (error) {
      throw new Error(`Hover failed: ${error.message}`);
    }
  }
}

export class PlaywrightFindElementTool extends BrowserTool {
  constructor(page) {
    super(
      "find_element",
      "Find elements on the page using various strategies",
      z.object({
        locator: z.object({
          role: z.string().optional().describe("ARIA role"),
          name: z.string().optional().describe("Accessible name"),
          text: z.string().optional().describe("Text content"),
          placeholder: z.string().optional().describe("Placeholder text"),
          selector: z.string().optional().describe("CSS selector")
        }).describe("Locator options"),
        reasoning: z.string().describe("Reasoning for this action")
      })
    );
    this.page = page;
  }

  async call({ locator, reasoning }) {
    try {
      let element;
      
      if (locator.role) {
        element = this.page.getByRole(locator.role, locator.name ? { name: locator.name } : {});
      } else if (locator.text) {
        element = this.page.getByText(locator.text);
      } else if (locator.placeholder) {
        element = this.page.getByPlaceholder(locator.placeholder);
      } else if (locator.selector) {
        element = this.page.locator(locator.selector);
      } else {
        throw new Error("At least one locator property required");
      }

      const count = await element.count();
      const elements = [];
      
      for (let i = 0; i < Math.min(count, 5); i++) {
        const el = element.nth(i);
        const text = await el.textContent().catch(() => '');
        const tagName = await el.evaluate(e => e.tagName.toLowerCase());
        elements.push({ index: i, tagName, text: text?.trim() || '' });
      }

      return `Found ${count} element(s): ${JSON.stringify(elements)}`;
    } catch (error) {
      throw new Error(`Find element failed: ${error.message}`);
    }
  }
}

export class PlaywrightHighlightElementsTool extends BrowserTool {
  constructor(page) {
    super(
      "highlight_elements",
      "Highlight interactive elements on the page with numbered overlays",
      z.object({
        elementTypes: z.array(z.string()).default(["button", "input", "a", "select"]).optional(),
        reasoning: z.string().describe("Reasoning for this action")
      })
    );
    this.page = page;
  }

  async call({ elementTypes = ["button", "input", "a", "select"], reasoning }) {
    try {
      const selector = elementTypes.join(', ');
      
      // Add highlight styles
      await this.page.addStyleTag({
        content: `
          .playwright-highlight {
            outline: 2px solid red !important;
            outline-offset: 2px !important;
            position: relative !important;
          }
          .playwright-highlight::before {
            content: attr(data-highlight-index) !important;
            position: absolute !important;
            top: -20px !important;
            left: -2px !important;
            background: red !important;
            color: white !important;
            padding: 2px 6px !important;
            font-size: 12px !important;
            font-weight: bold !important;
            z-index: 10000 !important;
          }
        `
      });

      // Highlight elements
      const count = await this.page.evaluate((sel) => {
        const elements = document.querySelectorAll(sel);
        let index = 0;
        elements.forEach(el => {
          if (el.offsetWidth > 0 && el.offsetHeight > 0) { // Only visible elements
            el.classList.add('playwright-highlight');
            el.setAttribute('data-highlight-index', String(index++));
          }
        });
        return index;
      }, selector);

      // Remove highlights after 5 seconds
      setTimeout(async () => {
        await this.page.evaluate(() => {
          document.querySelectorAll('.playwright-highlight').forEach(el => {
            el.classList.remove('playwright-highlight');
            el.removeAttribute('data-highlight-index');
          });
        }).catch(() => {}); // Ignore errors if page navigated
      }, 5000);

      return `Highlighted ${count} interactive elements`;
    } catch (error) {
      throw new Error(`Highlight failed: ${error.message}`);
    }
  }
}

export class PlaywrightGetDomStructureTool extends BrowserTool {
  constructor(page) {
    super(
      "get_dom_structure",
      "Get the DOM structure of the page",
      z.object({
        maxDepth: z.number().min(1).max(5).default(3).optional(),
        reasoning: z.string().describe("Reasoning for this action")
      })
    );
    this.page = page;
  }

  async call({ maxDepth = 3, reasoning }) {
    try {
      const structure = await this.page.evaluate((depth) => {
        function buildTree(element, currentDepth) {
          if (currentDepth > depth || !element) return null;
          
          const node = {
            tagName: element.tagName?.toLowerCase(),
            id: element.id || null,
            className: element.className || null,
            textContent: element.textContent?.trim().substring(0, 50) || null,
            children: []
          };
          
          for (const child of element.children) {
            const childNode = buildTree(child, currentDepth + 1);
            if (childNode) node.children.push(childNode);
          }
          
          return node;
        }
        
        return buildTree(document.body, 0);
      }, maxDepth);
      
      return JSON.stringify(structure, null, 2);
    } catch (error) {
      throw new Error(`Failed to get DOM structure: ${error.message}`);
    }
  }
}

export class PlaywrightVerifyActionTool extends BrowserTool {
  constructor(page) {
    super(
      "verify_action",
      "Verify that a previous action had the expected effect",
      z.object({
        checks: z.array(z.object({
          type: z.enum(["element_exists", "text_contains", "url_contains"]),
          value: z.string(),
          reasoning: z.string().describe("Reasoning for this action")
        })).describe("Verification checks to perform")
      })
    );
    this.page = page;
  }

  async call({ checks, reasoning }) {
    try {
      const results = [];
      
      for (const check of checks) {
        let passed = false;
        let detail = '';
        
        switch (check.type) {
          case 'element_exists':
            const count = await this.page.locator(check.value).count();
            passed = count > 0;
            detail = `Element ${check.value}: ${passed ? 'found' : 'not found'}`;
            break;
            
          case 'text_contains':
            const bodyText = await this.page.textContent('body');
            passed = bodyText.includes(check.value);
            detail = `Text "${check.value}": ${passed ? 'found' : 'not found'}`;
            break;
            
          case 'url_contains':
            const url = this.page.url();
            passed = url.includes(check.value);
            detail = `URL contains "${check.value}": ${passed ? 'yes' : 'no'} (current: ${url})`;
            break;
        }
        
        results.push({ check: check.type, passed, detail });
      }
      
      const allPassed = results.every(r => r.passed);
      const summary = results.map(r => r.detail).join('\n');
      
      return `Verification ${allPassed ? 'PASSED' : 'FAILED'}:\n${summary}`;
    } catch (error) {
      throw new Error(`Verification failed: ${error.message}`);
    }
  }
}

