import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

/**
 * Advanced Scroll Tool - Container-aware scrolling with element support
 * Inspired by browser-use's scroll action with container detection
 */
export class PlaywrightAdvancedScrollTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "advanced_scroll",
      "Advanced scrolling with container detection. Can scroll the page, specific elements, or scrollable containers like dropdowns.",
      z.object({
        direction: z.enum(["up", "down", "left", "right"])
          .optional()
          .default("down")
          .describe("Scroll direction"),
        pages: z.number()
          .optional()
          .default(1)
          .describe("Number of pages to scroll (0.5 for half page, 1 for full page, etc)"),
        selector: z.string()
          .optional()
          .describe("Optional selector for element to scroll within"),
        smooth: z.boolean()
          .optional()
          .default(true)
          .describe("Use smooth scrolling animation")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ direction = "down", pages = 1, selector, smooth = true }) {
    try {
      if (this.options.verbose) {
        console.log(`📜 Advanced scroll: ${direction} by ${pages} page(s)${selector ? ` in element: ${selector}` : ''}`);
      }

      // Get viewport dimensions
      const viewport = await this.page.evaluate(() => ({
        width: window.innerWidth,
        height: window.innerHeight
      }));

      // Calculate scroll amounts
      const horizontalAmount = Math.floor(viewport.width * pages);
      const verticalAmount = Math.floor(viewport.height * pages);

      let scrollX = 0;
      let scrollY = 0;

      switch (direction) {
        case "down":
          scrollY = verticalAmount;
          break;
        case "up":
          scrollY = -verticalAmount;
          break;
        case "right":
          scrollX = horizontalAmount;
          break;
        case "left":
          scrollX = -horizontalAmount;
          break;
      }

      // Element-specific or container scrolling
      if (selector) {
        const result = await this.scrollElement(selector, scrollX, scrollY, smooth);
        if (result.success) {
          return result.message;
        }
        // Fall back to page scrolling if element scrolling failed
        if (this.options.verbose) {
          console.log(`   Falling back to page scroll: ${result.reason}`);
        }
      }

      // Page-level scrolling
      const scrollResult = await this.scrollPage(scrollX, scrollY, smooth);
      return scrollResult.message;

    } catch (error) {
      const errorMsg = `Advanced scroll failed: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }

  async scrollElement(selector, scrollX, scrollY, smooth) {
    try {
      // JavaScript to handle element/container scrolling
      const result = await this.page.evaluate(({ selector, scrollX, scrollY, smooth }) => {
        // Find the target element
        let element;
        try {
          element = document.querySelector(selector);
        } catch (e) {
          return { 
            success: false, 
            reason: `Invalid selector: ${selector}` 
          };
        }

        if (!element) {
          return { 
            success: false, 
            reason: `Element not found: ${selector}` 
          };
        }

        // Function to find scrollable container
        const findScrollableContainer = (el) => {
          let current = el;
          let attempts = 0;
          
          while (current && attempts < 15) {
            const style = window.getComputedStyle(current);
            const overflowY = style.overflowY;
            const overflowX = style.overflowX;
            
            // Check if element is scrollable
            const canScrollY = overflowY === 'auto' || overflowY === 'scroll' || overflowY === 'overlay';
            const canScrollX = overflowX === 'auto' || overflowX === 'scroll' || overflowX === 'overlay';
            
            const hasVerticalScroll = current.scrollHeight > current.clientHeight;
            const hasHorizontalScroll = current.scrollWidth > current.clientWidth;
            
            if ((canScrollY && hasVerticalScroll && scrollY !== 0) || 
                (canScrollX && hasHorizontalScroll && scrollX !== 0)) {
              return current;
            }
            
            // Don't go beyond body/html
            if (current === document.body || current === document.documentElement) {
              break;
            }
            
            current = current.parentElement;
            attempts++;
          }
          
          return null;
        };

        // Try to find scrollable container
        const container = findScrollableContainer(element);
        
        if (container) {
          // Scroll the container
          const beforeScrollTop = container.scrollTop;
          const beforeScrollLeft = container.scrollLeft;
          
          if (smooth) {
            container.scrollBy({
              top: scrollY,
              left: scrollX,
              behavior: 'smooth'
            });
          } else {
            container.scrollTop += scrollY;
            container.scrollLeft += scrollX;
          }
          
          // Check if scroll actually happened
          // For smooth scroll, we can't immediately check, so assume success
          if (!smooth) {
            const actualScrollY = container.scrollTop - beforeScrollTop;
            const actualScrollX = container.scrollLeft - beforeScrollLeft;
            
            if (Math.abs(actualScrollY) < 1 && Math.abs(actualScrollX) < 1) {
              return {
                success: false,
                reason: 'Container found but scroll had no effect'
              };
            }
          }
          
          // Describe the container
          let containerDesc = container.tagName.toLowerCase();
          if (container.id) {
            containerDesc += `#${container.id}`;
          } else if (container.className) {
            containerDesc += `.${container.className.split(' ')[0]}`;
          }
          
          return {
            success: true,
            container: containerDesc,
            scrolled: { x: scrollX, y: scrollY }
          };
        }
        
        // No scrollable container found, try scrolling the element itself
        if (element.scrollHeight > element.clientHeight || element.scrollWidth > element.clientWidth) {
          const beforeScrollTop = element.scrollTop;
          const beforeScrollLeft = element.scrollLeft;
          
          if (smooth) {
            element.scrollBy({
              top: scrollY,
              left: scrollX,
              behavior: 'smooth'
            });
          } else {
            element.scrollTop += scrollY;
            element.scrollLeft += scrollX;
          }
          
          return {
            success: true,
            container: 'element itself',
            scrolled: { x: scrollX, y: scrollY }
          };
        }
        
        return {
          success: false,
          reason: 'No scrollable container found for element'
        };
      }, { selector, scrollX, scrollY, smooth });

      if (result.success) {
        const message = `Scrolled ${result.container} by ${Math.abs(scrollY)}px ${scrollY > 0 ? 'down' : scrollY < 0 ? 'up' : ''}${scrollX !== 0 ? ` and ${Math.abs(scrollX)}px ${scrollX > 0 ? 'right' : 'left'}` : ''}`;
        
        if (this.options.verbose) {
          console.log(`   ${message}`);
        }
        
        return { success: true, message };
      }

      return { success: false, reason: result.reason };

    } catch (error) {
      return { 
        success: false, 
        reason: `Element scroll error: ${error.message}` 
      };
    }
  }

  async scrollPage(scrollX, scrollY, smooth) {
    try {
      // Try smart scrolling first (finds appropriate container)
      const smartScrollResult = await this.page.evaluate(({ scrollX, scrollY, smooth }) => {
        // Try to find the main scrollable container
        const containers = [
          document.documentElement,
          document.body,
          document.querySelector('main'),
          document.querySelector('[role="main"]'),
          document.querySelector('.content'),
          document.querySelector('#content')
        ].filter(Boolean);

        for (const container of containers) {
          const beforeTop = container.scrollTop;
          const beforeLeft = container.scrollLeft;
          
          if (smooth) {
            container.scrollBy({
              top: scrollY,
              left: scrollX,
              behavior: 'smooth'
            });
          } else {
            container.scrollBy(scrollX, scrollY);
          }
          
          // For non-smooth, check if scroll worked
          if (!smooth) {
            const movedY = container.scrollTop - beforeTop;
            const movedX = container.scrollLeft - beforeLeft;
            if (Math.abs(movedY) > 0 || Math.abs(movedX) > 0) {
              return { 
                success: true, 
                container: container.tagName.toLowerCase() 
              };
            }
          } else {
            // For smooth scrolling, assume success if container is scrollable
            if (container.scrollHeight > container.clientHeight || 
                container.scrollWidth > container.clientWidth) {
              return { 
                success: true, 
                container: container.tagName.toLowerCase() 
              };
            }
          }
        }

        // Fallback to window scroll
        window.scrollBy(scrollX, scrollY);
        return { success: true, container: 'window' };
      }, { scrollX, scrollY, smooth });

      // Wait for content to settle
      await this.page.waitForTimeout(smooth ? 500 : 200);

      const direction = [];
      if (scrollY > 0) direction.push('down');
      if (scrollY < 0) direction.push('up');
      if (scrollX > 0) direction.push('right');
      if (scrollX < 0) direction.push('left');

      const message = `Scrolled ${smartScrollResult.container} ${direction.join(' and ')} by ${Math.abs(scrollY || scrollX)}px`;
      
      if (this.options.verbose) {
        console.log(`   ${message}`);
      }

      return { success: true, message };

    } catch (error) {
      // Hard fallback
      await this.page.evaluate((y) => window.scrollBy(0, y), scrollY);
      return { 
        success: true, 
        message: `Scrolled page ${scrollY > 0 ? 'down' : 'up'} by ${Math.abs(scrollY)}px (fallback)` 
      };
    }
  }
}

/**
 * Scroll to Bottom/Top Tool - Quickly navigate to page extremes
 */
export class PlaywrightScrollToExtremeTool extends BrowserTool {
  constructor(page, options = {}) {
    super(
      "scroll_to_extreme",
      "Scroll to the top or bottom of the page or element",
      z.object({
        position: z.enum(["top", "bottom"]).describe("Where to scroll to"),
        selector: z.string()
          .optional()
          .describe("Optional selector for element to scroll within"),
        smooth: z.boolean()
          .optional()
          .default(true)
          .describe("Use smooth scrolling")
      })
    );
    this.page = page;
    this.options = options;
  }

  async call({ position, selector, smooth = true }) {
    try {
      if (selector) {
        // Scroll within element
        await this.page.evaluate(({ selector, position, smooth }) => {
          const element = document.querySelector(selector);
          if (!element) throw new Error(`Element not found: ${selector}`);
          
          if (position === "top") {
            element.scrollTo({ top: 0, behavior: smooth ? 'smooth' : 'auto' });
          } else {
            element.scrollTo({ 
              top: element.scrollHeight, 
              behavior: smooth ? 'smooth' : 'auto' 
            });
          }
        }, { selector, position, smooth });
        
        return `Scrolled to ${position} of element: ${selector}`;
      } else {
        // Scroll page
        if (position === "top") {
          await this.page.evaluate((smooth) => {
            window.scrollTo({ top: 0, behavior: smooth ? 'smooth' : 'auto' });
          }, smooth);
        } else {
          await this.page.evaluate((smooth) => {
            window.scrollTo({ 
              top: document.documentElement.scrollHeight,
              behavior: smooth ? 'smooth' : 'auto'
            });
          }, smooth);
        }
        
        return `Scrolled to ${position} of page`;
      }
    } catch (error) {
      const errorMsg = `Failed to scroll to ${position}: ${error.message}`;
      console.error(`❌ ${errorMsg}`);
      throw new Error(errorMsg);
    }
  }
}