/**
 * Password Manager Integration for Vibe Extension
 * Uses chrome.passwordsPrivate API to access Chrome's password store
 */

export class PasswordManager {
  constructor() {
    this.isAvailable = typeof chrome !== 'undefined' && chrome.passwordsPrivate;
  }

  /**
   * Check if the password manager API is available
   */
  async checkAvailability() {
    if (!this.isAvailable) {
      throw new Error('chrome.passwordsPrivate API is not available');
    }
    
    // Test API access
    try {
      await this.getPasswordCheckStatus();
      return true;
    } catch (error) {
      console.error('Password manager API access denied:', error);
      return false;
    }
  }

  /**
   * Get saved passwords for a specific URL or all passwords
   * @param {string} url - Optional URL to filter passwords
   * @returns {Promise<Array>} List of password entries
   */
  async getSavedPasswords(url = null) {
    return new Promise((resolve, reject) => {
      chrome.passwordsPrivate.getSavedPasswordList((passwords) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
          return;
        }
        
        if (url) {
          // Filter passwords by URL
          const filtered = passwords.filter(entry => 
            entry.affiliatedDomains.some(domain => 
              domain.url.includes(url) || domain.signonRealm.includes(url)
            )
          );
          resolve(filtered);
        } else {
          resolve(passwords);
        }
      });
    });
  }

  /**
   * Get plaintext password for a specific entry
   * @param {number} id - Password entry ID
   * @param {string} reason - Reason for accessing (VIEW, COPY, EDIT)
   * @returns {Promise<string>} The plaintext password
   */
  async getPassword(id, reason = 'VIEW') {
    return new Promise((resolve, reject) => {
      chrome.passwordsPrivate.requestPlaintextPassword(
        id,
        chrome.passwordsPrivate.PlaintextReason[reason],
        (password) => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
            return;
          }
          resolve(password);
        }
      );
    });
  }

  /**
   * Add a new password entry
   * @param {Object} options - Password details
   * @returns {Promise<void>}
   */
  async addPassword(options) {
    const { url, username, password, note = '', useAccountStore = false } = options;
    
    return new Promise((resolve, reject) => {
      chrome.passwordsPrivate.addPassword(
        {
          url,
          username,
          password,
          note,
          useAccountStore
        },
        () => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
            return;
          }
          resolve();
        }
      );
    });
  }

  /**
   * Update an existing password
   * @param {Object} credential - Updated credential info
   * @returns {Promise<void>}
   */
  async updatePassword(credential) {
    return new Promise((resolve, reject) => {
      chrome.passwordsPrivate.changeCredential(credential, () => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
          return;
        }
        resolve();
      });
    });
  }

  /**
   * Remove a password entry
   * @param {number} id - Password entry ID
   * @param {string} fromStores - Which store to remove from (DEVICE, ACCOUNT, DEVICE_AND_ACCOUNT)
   * @returns {Promise<void>}
   */
  async removePassword(id, fromStores = 'DEVICE') {
    return new Promise((resolve, reject) => {
      chrome.passwordsPrivate.removeCredential(
        id,
        chrome.passwordsPrivate.PasswordStoreSet[fromStores],
        () => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
            return;
          }
          resolve();
        }
      );
    });
  }

  /**
   * Get password check status
   * @returns {Promise<Object>} Password check status
   */
  async getPasswordCheckStatus() {
    return new Promise((resolve, reject) => {
      chrome.passwordsPrivate.getPasswordCheckStatus((status) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
          return;
        }
        resolve(status);
      });
    });
  }

  /**
   * Get credentials for a specific service (used by SecretsManager tool)
   * @param {string} service - Service name (e.g., 'LinkedIn', 'Gmail')
   * @returns {Promise<Object>} Credentials object with login and password
   */
  async getCredentialsForService(service) {
    const passwords = await this.getSavedPasswords();
    
    // Find passwords that match the service name
    const matches = passwords.filter(entry => {
      const serviceLower = service.toLowerCase();
      return entry.affiliatedDomains.some(domain => 
        domain.name.toLowerCase().includes(serviceLower) ||
        domain.url.toLowerCase().includes(serviceLower)
      );
    });

    if (matches.length === 0) {
      throw new Error(`No credentials found for service: ${service}`);
    }

    // Get the first match
    const match = matches[0];
    const password = await this.getPassword(match.id, 'VIEW');

    return {
      service,
      login: match.username,
      password,
      url: match.affiliatedDomains[0].url
    };
  }

  /**
   * Authenticate user before accessing passwords
   * @returns {Promise<boolean>} Authentication result
   */
  async authenticate() {
    return new Promise((resolve, reject) => {
      chrome.passwordsPrivate.authenticate((result) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
          return;
        }
        resolve(result);
      });
    });
  }
}

// Create singleton instance
export const passwordManager = new PasswordManager();