/**
 * MCP (Model Context Protocol) Types for Vibe AI Browser Extension
 * 
 * Only supports remote MCP servers since Chrome extensions cannot spawn local processes.
 */

import { z } from 'zod';

/**
 * OAuth configuration for MCP servers
 */
export const McpOAuthConfigSchema = z.object({
  clientId: z.string().optional().describe('OAuth client ID'),
  clientSecret: z.string().optional().describe('OAuth client secret'),
  scope: z.string().optional().describe('OAuth scopes to request'),
});
export type McpOAuthConfig = z.infer<typeof McpOAuthConfigSchema>;

/**
 * Remote MCP server configuration
 * Chrome extensions only support remote servers (SSE/StreamableHTTP)
 */
export const McpServerConfigSchema = z.object({
  name: z.string().describe('Unique name for this MCP server'),
  url: z.string().url().describe('URL of the remote MCP server'),
  enabled: z.boolean().default(true).describe('Whether this server is enabled'),
  headers: z.record(z.string(), z.string()).optional().describe('Custom headers to send'),
  oauth: z.union([McpOAuthConfigSchema, z.literal(false)]).optional()
    .describe('OAuth configuration or false to disable'),
  timeout: z.number().int().positive().default(30000)
    .describe('Connection timeout in milliseconds'),
});
export type McpServerConfig = z.infer<typeof McpServerConfigSchema>;

/**
 * MCP configuration stored in extension storage
 */
export const McpConfigSchema = z.object({
  servers: z.record(z.string(), McpServerConfigSchema).default({}),
});
export type McpConfig = z.infer<typeof McpConfigSchema>;

/**
 * MCP server connection status
 */
export type McpServerStatus = 
  | { status: 'connected' }
  | { status: 'disabled' }
  | { status: 'connecting' }
  | { status: 'failed'; error: string }
  | { status: 'needs_auth' };

/**
 * MCP tool definition from server
 */
export interface McpToolDefinition {
  name: string;
  description?: string;
  inputSchema: {
    type: 'object';
    properties?: Record<string, unknown>;
    required?: string[];
  };
}

/**
 * MCP tool call result
 */
export interface McpToolResult {
  content: Array<{
    type: 'text' | 'image' | 'resource';
    text?: string;
    data?: string;
    mimeType?: string;
  }>;
  isError?: boolean;
}

/**
 * OAuth tokens stored for MCP servers
 */
export const McpOAuthTokensSchema = z.object({
  accessToken: z.string(),
  refreshToken: z.string().optional(),
  expiresAt: z.number().optional(),
  scope: z.string().optional(),
});
export type McpOAuthTokens = z.infer<typeof McpOAuthTokensSchema>;

/**
 * OAuth auth entry stored per server
 */
export const McpAuthEntrySchema = z.object({
  tokens: McpOAuthTokensSchema.optional(),
  serverUrl: z.string().optional(),
});
export type McpAuthEntry = z.infer<typeof McpAuthEntrySchema>;
