/**
 * MCP Server Types for Vibe Extension
 * 
 * Types for exposing Vibe as an MCP server that external clients can connect to.
 */

import { z } from 'zod';

/**
 * MCP Protocol version
 */
export const MCP_PROTOCOL_VERSION = '2024-11-05';

/**
 * JSON-RPC request structure
 */
export interface JsonRpcRequest {
  jsonrpc: '2.0';
  id?: number | string;
  method: string;
  params?: Record<string, unknown>;
}

/**
 * JSON-RPC response structure
 */
export interface JsonRpcResponse {
  jsonrpc: '2.0';
  id?: number | string;
  result?: unknown;
  error?: JsonRpcError;
}

/**
 * JSON-RPC error structure
 */
export interface JsonRpcError {
  code: number;
  message: string;
  data?: unknown;
}

/**
 * Standard JSON-RPC error codes
 */
export const JsonRpcErrorCodes = {
  PARSE_ERROR: -32700,
  INVALID_REQUEST: -32600,
  METHOD_NOT_FOUND: -32601,
  INVALID_PARAMS: -32602,
  INTERNAL_ERROR: -32603,
} as const;

/**
 * MCP Initialize request params
 */
export interface InitializeParams {
  protocolVersion: string;
  capabilities: {
    roots?: { listChanged?: boolean };
    sampling?: Record<string, unknown>;
  };
  clientInfo: {
    name: string;
    version: string;
  };
}

/**
 * MCP Initialize response result
 */
export interface InitializeResult {
  protocolVersion: string;
  capabilities: {
    tools?: { listChanged?: boolean };
    resources?: { subscribe?: boolean; listChanged?: boolean };
    prompts?: { listChanged?: boolean };
    logging?: Record<string, unknown>;
  };
  serverInfo: {
    name: string;
    version: string;
  };
}

/**
 * MCP List Tools result
 */
export interface ListToolsResult {
  tools: McpServerToolDefinition[];
}

/**
 * Tool definition exposed by server
 */
export interface McpServerToolDefinition {
  name: string;
  description?: string;
  inputSchema: {
    type: 'object';
    properties?: Record<string, JsonSchemaProperty>;
    required?: string[];
  };
}

/**
 * JSON Schema property for tool parameters
 */
export interface JsonSchemaProperty {
  type?: string;
  description?: string;
  enum?: string[];
  items?: JsonSchemaProperty;
  properties?: Record<string, JsonSchemaProperty>;
  required?: string[];
  default?: unknown;
}

/**
 * MCP Call Tool request params
 */
export interface CallToolParams {
  name: string;
  arguments?: Record<string, unknown>;
}

/**
 * MCP Call Tool result
 */
export interface CallToolResult {
  content: ToolResultContent[];
  isError?: boolean;
}

/**
 * Tool result content item
 */
export type ToolResultContent = 
  | { type: 'text'; text: string }
  | { type: 'image'; data: string; mimeType: string }
  | { type: 'resource'; resource: { uri: string; text?: string; blob?: string; mimeType?: string } };

/**
 * MCP Server configuration
 */
export const McpServerSettingsSchema = z.object({
  enabled: z.boolean().default(false).describe('Whether MCP server is enabled'),
  port: z.number().int().min(1024).max(65535).default(19989)
    .describe('Port to listen on'),
  host: z.string().default('127.0.0.1')
    .describe('Host to bind to (127.0.0.1 for local only)'),
  token: z.string().optional()
    .describe('Optional authentication token'),
  exposedTools: z.array(z.string()).optional()
    .describe('List of tool names to expose (all if not specified)'),
});
export type McpServerSettings = z.infer<typeof McpServerSettingsSchema>;

/**
 * Connected client information
 */
export interface ConnectedClient {
  id: string;
  connectedAt: number;
  clientInfo?: {
    name: string;
    version: string;
  };
  lastActivity: number;
}

/**
 * MCP Server state
 */
export interface McpServerState {
  running: boolean;
  port: number;
  host: string;
  clients: ConnectedClient[];
  startedAt?: number;
  error?: string;
}

/**
 * Message from MCP client via Native Messaging
 */
export interface NativeMessage {
  type: 'client_connected' | 'client_disconnected' | 'client_message' | 'server_started' | 'server_stopped' | 'error';
  clientId?: string;
  message?: JsonRpcRequest;
  error?: string;
  port?: number;
}

/**
 * Message to MCP server via Native Messaging
 */
export interface NativeCommand {
  type: 'start' | 'stop' | 'send_response';
  clientId?: string;
  response?: JsonRpcResponse;
  settings?: McpServerSettings;
}
