/**
 * Tool Adapter for MCP Server
 * 
 * Converts Vibe's BrowserTools to MCP tool definitions and handles tool execution.
 */

import type { ZodType, ZodObject, ZodRawShape } from 'zod';
import type { 
  McpServerToolDefinition, 
  JsonSchemaProperty,
  CallToolResult 
} from './types';

// Type for BrowserTool - imported dynamically to avoid circular dependencies
interface BrowserToolInterface {
  name: string;
  description: string;
  schema: ZodType;
  call(args: Record<string, unknown>): Promise<string>;
}

// Internal type for Zod schema internals (not in public API but needed for conversion)
interface ZodDef {
  typeName?: string;
  description?: string;
  innerType?: ZodType;
  type?: ZodType;
  values?: string[];
  value?: unknown;
  options?: ZodType[];
  defaultValue?: () => unknown;
}

/**
 * Get Zod definition with proper typing
 */
function getZodDef(schema: ZodType): ZodDef {
   
  return (schema as any)._def || {};
}

/**
 * Get shape from ZodObject
 */
function getShape(schema: ZodType): Record<string, ZodType> {
   
  return (schema as any).shape || {};
}

/**
 * Convert a Zod schema to JSON Schema format
 */
function zodToJsonSchema(schema: ZodType): { 
  type: 'object'; 
  properties: Record<string, JsonSchemaProperty>;
  required?: string[];
} {
  const def = getZodDef(schema);
  
  // Handle ZodObject
  if (def.typeName === 'ZodObject') {
    const shape = getShape(schema);
    const properties: Record<string, JsonSchemaProperty> = {};
    const required: string[] = [];
    
    for (const [key, value] of Object.entries(shape)) {
      properties[key] = zodPropertyToJsonSchema(value);
      
      // Check if field is required (not optional)
      if (!isOptional(value)) {
        required.push(key);
      }
    }
    
    return {
      type: 'object',
      properties,
      ...(required.length > 0 ? { required } : {}),
    };
  }
  
  // Default to empty object schema
  return { type: 'object', properties: {} };
}

/**
 * Convert a single Zod property to JSON Schema
 */
function zodPropertyToJsonSchema(schema: ZodType): JsonSchemaProperty {
  const def = getZodDef(schema);
  const typeName = def.typeName;
  
  // Get description if available
  const description = def.description;
  
  // Handle optional wrapper
  if (typeName === 'ZodOptional' && def.innerType) {
    const innerSchema = zodPropertyToJsonSchema(def.innerType);
    return { ...innerSchema, description: description || innerSchema.description };
  }
  
  // Handle nullable wrapper
  if (typeName === 'ZodNullable' && def.innerType) {
    const innerSchema = zodPropertyToJsonSchema(def.innerType);
    return { ...innerSchema, description: description || innerSchema.description };
  }
  
  // Handle default wrapper
  if (typeName === 'ZodDefault' && def.innerType) {
    const innerSchema = zodPropertyToJsonSchema(def.innerType);
    return { 
      ...innerSchema, 
      default: def.defaultValue?.(),
      description: description || innerSchema.description 
    };
  }
  
  // Handle basic types
  switch (typeName) {
    case 'ZodString':
      return { type: 'string', description };
      
    case 'ZodNumber':
      return { type: 'number', description };
      
    case 'ZodBoolean':
      return { type: 'boolean', description };
      
    case 'ZodEnum':
      return { 
        type: 'string', 
        enum: def.values,
        description 
      };
      
    case 'ZodArray':
      return {
        type: 'array',
        items: def.type ? zodPropertyToJsonSchema(def.type) : undefined,
        description,
      };
      
    case 'ZodObject': {
      const shape = getShape(schema);
      const properties: Record<string, JsonSchemaProperty> = {};
      const required: string[] = [];
      
      for (const [key, value] of Object.entries(shape)) {
        properties[key] = zodPropertyToJsonSchema(value);
        if (!isOptional(value)) {
          required.push(key);
        }
      }
      
      return {
        type: 'object',
        properties,
        ...(required.length > 0 ? { required } : {}),
        description,
      };
    }
    
    case 'ZodLiteral':
      return { 
        type: typeof def.value as string,
        enum: def.value !== undefined ? [def.value as string] : undefined,
        description 
      };
      
    case 'ZodUnion': {
      // For unions, try to extract enum values if all are literals
      const options = (def.options || []) as ZodType[];
      const literals = options.filter(o => getZodDef(o).typeName === 'ZodLiteral');
      if (literals.length === options.length && literals.length > 0) {
        const values = literals.map(l => getZodDef(l).value as string);
        return { type: 'string', enum: values, description };
      }
      // Fall back to any
      return { description };
    }
      
    default:
      // Unknown type - return with just description
      return { description };
  }
}

/**
 * Check if a Zod type is optional
 */
function isOptional(schema: ZodType): boolean {
  const typeName = getZodDef(schema).typeName;
  if (typeName === 'ZodOptional') return true;
  if (typeName === 'ZodNullable') return true;
  if (typeName === 'ZodDefault') return true;
  return false;
}

/**
 * Convert a BrowserTool to MCP tool definition
 */
export function browserToolToMcp(tool: BrowserToolInterface): McpServerToolDefinition {
  return {
    name: tool.name,
    description: tool.description,
    inputSchema: zodToJsonSchema(tool.schema),
  };
}

/**
 * Execute a BrowserTool and return MCP-formatted result
 */
export async function executeBrowserTool(
  tool: BrowserToolInterface, 
  args: Record<string, unknown>
): Promise<CallToolResult> {
  try {
    const result = await tool.call(args);
    
    // Check if result looks like a base64 image
    if (typeof result === 'string' && result.startsWith('data:image/')) {
      const [header, data] = result.split(',');
      const mimeMatch = header.match(/data:(image\/[^;]+)/);
      const mimeType = mimeMatch ? mimeMatch[1] : 'image/png';
      
      return {
        content: [{ type: 'image', data, mimeType }],
      };
    }
    
    // Check if result is JSON
    if (typeof result === 'string') {
      try {
        const parsed = JSON.parse(result);
        // If it parsed successfully, still return as text but formatted
        return {
          content: [{ type: 'text', text: JSON.stringify(parsed, null, 2) }],
        };
      } catch {
        // Not JSON, return as plain text
        return {
          content: [{ type: 'text', text: result }],
        };
      }
    }
    
    // Non-string result
    return {
      content: [{ type: 'text', text: JSON.stringify(result, null, 2) }],
    };
    
  } catch (error) {
    const message = error instanceof Error ? error.message : String(error);
    return {
      content: [{ type: 'text', text: `Error: ${message}` }],
      isError: true,
    };
  }
}

/**
 * Tool Registry - manages available tools for MCP server
 */
export class ToolRegistry {
  private tools: Map<string, BrowserToolInterface> = new Map();
  private toolDefinitions: Map<string, McpServerToolDefinition> = new Map();
  
  /**
   * Register a tool
   */
  register(tool: BrowserToolInterface): void {
    this.tools.set(tool.name, tool);
    this.toolDefinitions.set(tool.name, browserToolToMcp(tool));
  }
  
  /**
   * Register multiple tools
   */
  registerAll(tools: BrowserToolInterface[]): void {
    for (const tool of tools) {
      this.register(tool);
    }
  }
  
  /**
   * Get all tool definitions
   */
  getDefinitions(): McpServerToolDefinition[] {
    return Array.from(this.toolDefinitions.values());
  }
  
  /**
   * Get a specific tool by name
   */
  getTool(name: string): BrowserToolInterface | undefined {
    return this.tools.get(name);
  }
  
  /**
   * Check if tool exists
   */
  hasTool(name: string): boolean {
    return this.tools.has(name);
  }
  
  /**
   * Execute a tool by name
   */
  async execute(name: string, args: Record<string, unknown>): Promise<CallToolResult> {
    const tool = this.tools.get(name);
    if (!tool) {
      return {
        content: [{ type: 'text', text: `Tool not found: ${name}` }],
        isError: true,
      };
    }
    
    return executeBrowserTool(tool, args);
  }
  
  /**
   * Get list of tool names
   */
  getToolNames(): string[] {
    return Array.from(this.tools.keys());
  }
  
  /**
   * Clear all tools
   */
  clear(): void {
    this.tools.clear();
    this.toolDefinitions.clear();
  }
}

// Singleton instance
let registryInstance: ToolRegistry | null = null;
let toolsRegistered = false;

/**
 * Get the tool registry singleton
 */
export function getToolRegistry(): ToolRegistry {
  if (!registryInstance) {
    registryInstance = new ToolRegistry();
  }
  return registryInstance;
}

/**
 * Register all browser tools with the registry
 * This should be called once during extension initialization
 */
export async function registerBrowserTools(): Promise<void> {
  if (toolsRegistered) {
    return;
  }
  
  try {
    // Dynamically import browser tools to avoid circular dependencies
    // @ts-expect-error - JS module without type declarations
    const module = await import('../../extension/tools/BrowserToolsIndex.js');
    const browserTools = module.browserTools as BrowserToolInterface[];
    
    const registry = getToolRegistry();
    registry.registerAll(browserTools);
    toolsRegistered = true;
    
    console.log(`[MCP Server] Registered ${browserTools.length} browser tools`);
  } catch (error) {
    console.error('[MCP Server] Failed to register browser tools:', error);
  }
}
