/**
 * MCP Protocol Handler
 * 
 * Handles MCP JSON-RPC requests and routes them to appropriate handlers.
 */

import {
  JsonRpcRequest,
  JsonRpcResponse,
  JsonRpcErrorCodes,
  MCP_PROTOCOL_VERSION,
  InitializeParams,
  InitializeResult,
  ListToolsResult,
  CallToolParams,
  CallToolResult,
  ConnectedClient,
} from './types';
import { ToolRegistry, getToolRegistry } from './tool-adapter';

/**
 * MCP Protocol Handler
 */
export class McpProtocolHandler {
  private toolRegistry: ToolRegistry;
  private serverName: string;
  private serverVersion: string;
  private initialized: Map<string, boolean> = new Map();
  
  constructor(serverName = 'Vibe AI Browser', serverVersion = '1.0.0') {
    this.toolRegistry = getToolRegistry();
    this.serverName = serverName;
    this.serverVersion = serverVersion;
  }
  
  /**
   * Handle an incoming JSON-RPC request
   */
  async handleRequest(request: JsonRpcRequest, clientId: string): Promise<JsonRpcResponse> {
    const response: JsonRpcResponse = {
      jsonrpc: '2.0',
      id: request.id,
    };
    
    try {
      switch (request.method) {
        case 'initialize':
          response.result = await this.handleInitialize(
            request.params as unknown as InitializeParams,
            clientId
          );
          break;
          
        case 'notifications/initialized':
          // Client acknowledgment - no response needed for notifications
          // But we include result for non-notification calls
          if (request.id !== undefined) {
            response.result = {};
          }
          break;
          
        case 'tools/list':
          this.ensureInitialized(clientId);
          response.result = await this.handleListTools();
          break;
          
        case 'tools/call':
          this.ensureInitialized(clientId);
          response.result = await this.handleCallTool(
            request.params as unknown as CallToolParams
          );
          break;
          
        case 'ping':
          response.result = { pong: true };
          break;
          
        case 'resources/list':
          // Resources not supported yet
          response.result = { resources: [] };
          break;
          
        case 'prompts/list':
          // Prompts not supported yet
          response.result = { prompts: [] };
          break;
          
        default:
          // Check if it's a notification (no id)
          if (request.id === undefined) {
            // Notifications don't get responses
            return response;
          }
          
          response.error = {
            code: JsonRpcErrorCodes.METHOD_NOT_FOUND,
            message: `Method not found: ${request.method}`,
          };
      }
    } catch (error) {
      const message = error instanceof Error ? error.message : String(error);
      response.error = {
        code: JsonRpcErrorCodes.INTERNAL_ERROR,
        message,
      };
    }
    
    return response;
  }
  
  /**
   * Handle initialize request
   */
  private async handleInitialize(
    params: InitializeParams,
    clientId: string
  ): Promise<InitializeResult> {
    console.log('[MCP Server] Initialize request from:', params.clientInfo);
    
    this.initialized.set(clientId, true);
    
    return {
      protocolVersion: MCP_PROTOCOL_VERSION,
      capabilities: {
        tools: { listChanged: false },
        resources: { subscribe: false, listChanged: false },
        prompts: { listChanged: false },
      },
      serverInfo: {
        name: this.serverName,
        version: this.serverVersion,
      },
    };
  }
  
  /**
   * Handle list tools request
   */
  private async handleListTools(): Promise<ListToolsResult> {
    const tools = this.toolRegistry.getDefinitions();
    console.log('[MCP Server] Listing', tools.length, 'tools');
    return { tools };
  }
  
  /**
   * Handle call tool request
   */
  private async handleCallTool(params: CallToolParams): Promise<CallToolResult> {
    console.log('[MCP Server] Calling tool:', params.name);
    
    const result = await this.toolRegistry.execute(
      params.name,
      params.arguments || {}
    );
    
    if (result.isError) {
      console.error('[MCP Server] Tool error:', result.content);
    }
    
    return result;
  }
  
  /**
   * Ensure client is initialized
   */
  private ensureInitialized(clientId: string): void {
    if (!this.initialized.get(clientId)) {
      throw new Error('Client not initialized. Call initialize first.');
    }
  }
  
  /**
   * Handle client disconnect
   */
  onClientDisconnect(clientId: string): void {
    this.initialized.delete(clientId);
    console.log('[MCP Server] Client disconnected:', clientId);
  }
  
  /**
   * Get tool registry for registration
   */
  getToolRegistry(): ToolRegistry {
    return this.toolRegistry;
  }
}

// Singleton instance
let handlerInstance: McpProtocolHandler | null = null;

/**
 * Get the protocol handler singleton
 */
export function getProtocolHandler(): McpProtocolHandler {
  if (!handlerInstance) {
    handlerInstance = new McpProtocolHandler();
  }
  return handlerInstance;
}
